/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental;

import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.RateLimiter;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.patterns.PolledMeter;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.eventbus.Message;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.indexing.incremental.TypeIndexerFactory.TypeIndexer;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore.ContainerSyncState;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore.DirtynessMarker;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.utils.ThrottleAccumulator;
import net.bluemind.lifecycle.helper.SoftReset;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;
import net.bluemind.repository.provider.RepositoryProvider;

public class IncrementalIndexer extends AbstractVerticle {
	private static final Logger logger = LoggerFactory.getLogger(IncrementalIndexer.class);
	private final AtomicBoolean running = new AtomicBoolean(true);
	private final Map<String, TypeIndexerFactory> indexers = new HashMap<>();
	private final int concurrency;
	private final Set<String> inProgressUids;
	private final Semaphore maxConcurrent;
	private final Lock lock = new ReentrantLock();
	private final Condition wakeupCondition = lock.newCondition();

	// for junit
	public static final AtomicLong GEN = new AtomicLong();
	private static final Registry reg = MetricsRegistry.get();
	private static final IdFactory id = new IdFactory("incremental.indexing", reg, IncrementalIndexer.class);
	private static final Set<String> pendingIndexing = ConcurrentHashMap.newKeySet();
	private static final List<Thread> runningThreads = Collections.synchronizedList(new LinkedList<Thread>());
	private static final RateLimiter logConcurrencyRateLimiter = RateLimiter.create(5.0);

	static {
		PolledMeter.using(reg).withId(id.name("queue-size")).monitorValue(pendingIndexing, pi -> (double) pi.size());
		PolledMeter.using(reg).withId(id.name("running-threads")).monitorValue(runningThreads,
				pi -> (double) pi.size());
	}

	public IncrementalIndexer() {
		concurrency = CoreConfig.get().getInt(CoreConfig.Indexing.CONCURRENCY);
		maxConcurrent = new Semaphore(concurrency);
		inProgressUids = ConcurrentHashMap.newKeySet(concurrency);
		SoftReset.register(() -> {
			runningThreads.forEach(t -> {
				t.interrupt();
				try {
					t.join();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			});
			BmContext ctx = rootCtx();
			if (ctx.getDataSource() != null) {
				try {
					RepositoryProvider.instance(IIncrementalIndexingStore.class, rootCtx()).forgetStates();
				} catch (Exception e) {
					// fine, the table might not exist
				}
			}
			inProgressUids.clear();
			pendingIndexing.clear();
		});
	}

	@Override
	public void start() throws Exception {
		RunnableExtensionLoader<TypeIndexerFactory> rel = new RunnableExtensionLoader<>();
		List<TypeIndexerFactory> factos = rel.loadExtensions("net.bluemind.indexing.incremental", "indexer", "indexer",
				"impl");
		factos.stream().forEach(tf -> indexers.put(tf.type(), tf));
		logger.info("Available indexers: {}", indexers.keySet());
		logger.info("Indexing concurrency: {}", concurrency);

		ThrottleAccumulator<String> ta = new ThrottleAccumulator<>(vertx, Duration.ofSeconds(5), Message::body,
				message -> {
					ContainerUid cid = ContainerUid.of(message.body());
					pendingIndexing.add(cid.value());
					try {
						RepositoryProvider.instance(IIncrementalIndexingStore.class, rootCtx()).markDirty(cid);
						wakeupLoop();
					} catch (SQLException e) {
						logger.error("Unable to mark container {} as dirty: {}", cid, e.getMessage());
					}
				});
		vertx.eventBus().consumer("index.dirty", ta::handle);

		Thread mainLoopThread = Thread.ofVirtual().name("incremental-indexer-loop").unstarted(() -> {
			while (running.get()) {
				runningThreads.removeIf(thread -> !thread.isAlive());
				lock.lock();
				try {
					wakeupCondition.await(10, TimeUnit.MINUTES); // NOSONAR: fallback poll every 10 minutes
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					logger.warn("interrupted");
					running.set(false);
					continue;
				} finally {
					lock.unlock();
				}

				int limit = concurrency - inProgressUids.size();
				if (limit <= 0) {
					if (logConcurrencyRateLimiter.tryAcquire()) {
						logger.warn("maximum concurrency reached, retry later");
					}
					continue;
				}

				IIncrementalIndexingStore idxStore = RepositoryProvider.instance(IIncrementalIndexingStore.class,
						rootCtx());
				List<DirtynessMarker> dirties;
				try {
					dirties = idxStore.fetchNexts(inProgressUids, limit);
					wakeupLoop();
				} catch (Throwable e) {
					logger.error("fetchNexts failed: {}", e.getMessage());
					continue;
				}
				inProgressUids.addAll(dirties.stream().map(d -> d.container().value()).toList());
				dirties.stream().forEach(dirty -> {
					try {
						logger.debug("Waiting to acquire a thread slot...");
						maxConcurrent.acquire();
						Thread vt = getProcessingThread(idxStore, dirty);
						vt.start();
						runningThreads.add(vt);
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
						running.set(false);
					}
				});
			}
			logger.info("Stopped incremental indexer loop. Running: {}", running);
		});

		mainLoopThread.setUncaughtExceptionHandler((t, e) -> {
			logger.error("Uncaught exception in {}", t, e);
		});
		mainLoopThread.start();
	}

	private Thread getProcessingThread(IIncrementalIndexingStore idxStore, DirtynessMarker dirty) {
		Thread vt = Thread.ofVirtual().name("incremental-index-" + dirty.container().value()).unstarted(() -> {
			boolean pendingFlushed = false;
			try {
				pendingFlushed = incrementalIndex(rootCtx(), idxStore, dirty);
			} catch (Throwable e) {
				logger.error("Incremental indexing of {} failed: {}", dirty, e.getMessage());
			} finally {
				inProgressUids.remove(dirty.container().value());
				// if pendingFlushed is false, it means we have received an updated version
				// between flushes. We then retry as soon as possible
				if (pendingFlushed) {
					pendingIndexing.remove(dirty.container().value());
				}
				maxConcurrent.release();
				wakeupLoop();
			}
		});
		vt.setUncaughtExceptionHandler((t, e) -> {
			logger.error("uncaught exception while indexing {} using {}", dirty, idxStore, e);
		});
		return vt;
	}

	private void wakeupLoop() {
		lock.lock();
		try {
			wakeupCondition.signal();
		} finally {
			lock.unlock();
		}
	}

	@Override
	public void stop() {
		running.set(false);
	}

	private BmContext rootCtx() {
		return ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
	}

	private boolean incrementalIndex(BmContext ctx, IIncrementalIndexingStore idxStore, DirtynessMarker dirty)
			throws SQLException {
		long indexingLag = ChronoUnit.MILLIS.between(Instant.now(), dirty.since());
		if (indexingLag > 500) {
			logger.warn("Indexing lag is {}ms.", indexingLag);
		}
		ContainerSyncState state = idxStore.getState(dirty.container());

		IContainerRouteStore router = RepositoryProvider.instance(IContainerRouteStore.class, ctx);
		DataLocation loc = router.routeOf(dirty.container());
		if (loc == null) {
			idxStore.checkpointSync(state, dirty);
			return true;
		}
		IContainerStore store = RepositoryProvider.instance(IContainerStore.class, ctx, loc);
		Container cont = store.get(dirty.container().value());
		if (cont == null) {
			idxStore.checkpointSync(state, dirty);
			return true;
		}

		TypeIndexerFactory forType = indexers.get(cont.type);
		if (forType == null) {
			idxStore.checkpointSync(state, dirty);
			return true;
		}

		TypeIndexer indexer = forType.create(ctx, cont);
		long newVersion = indexer.indexDelta(state);
		boolean flushed = idxStore.checkpointSync(new ContainerSyncState(state.cont(), newVersion), dirty);
		GEN.incrementAndGet();
		return flushed;
	}

	// For Junit only
	public static int queuedIndexing() {
		return pendingIndexing.size();
	}

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {

		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new IncrementalIndexer();
		}

	}
}
