/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental;

import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;

import net.bluemind.core.container.api.IChangelogSupport;
import net.bluemind.core.container.api.IGetSupport;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore.ContainerSyncState;

public abstract class AbstractTypeIndexerByUid<T> implements TypeIndexerFactory.TypeIndexer {

	private static final Logger logger = LoggerFactory.getLogger(AbstractTypeIndexerByUid.class);

	public interface UidIndexWriter<W> {

		void write(ItemValue<W> toIndex);

		void deleteByUid(String id);

	}

	private final IChangelogSupport clSupport;
	private final UidIndexWriter<T> indexer;
	private final IGetSupport<T> crud;

	protected AbstractTypeIndexerByUid(IChangelogSupport clSupport, IGetSupport<T> crud, UidIndexWriter<T> indexer) {
		this.clSupport = clSupport;
		this.crud = crud;
		this.indexer = indexer;
	}

	@Override
	public long indexDelta(ContainerSyncState startState) {
		if (startState.version() == IIncrementalIndexingStore.DO_NOT_INDEX) {
			return startState.version();
		}

		try {
			var chrono = Stopwatch.createStarted();
			ContainerChangeset<String> changeset = clSupport.changeset(startState.version());

			for (String addedOrUpdated : Iterables.concat(changeset.created, changeset.updated)) {
				indexer.write(crud.getComplete(addedOrUpdated));
			}
			for (String deleted : changeset.deleted) {
				indexer.deleteByUid(deleted);
			}
			if (logger.isInfoEnabled() && changeset.version > startState.version()) {
				logger.info("[{}] incr update from v{} to v{}  took {}ms (c: {}, u: {}, d: {})",
						startState.cont().value(), startState.version(), changeset.version,
						chrono.elapsed(TimeUnit.MILLISECONDS), changeset.created.size(), changeset.updated.size(),
						changeset.deleted.size());
			}
			return changeset.version;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return startState.version();
		}
	}

}
