/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.indexing.incremental.calendar;

import java.util.List;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.calendar.persistence.VEventIndexStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexer;
import net.bluemind.indexing.incremental.AbstractTypeIndexer.ICrudFactory;
import net.bluemind.indexing.incremental.AbstractTypeIndexer.IndexWriter;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.lib.elasticsearch.ESearchActivator;

public class VEventsIncrementalIndexFactory implements TypeIndexerFactory {

	public String type() {
		return ICalendarUids.TYPE;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		ICrudFactory<VEventSeries> calendarApi = () -> ctx.provider().instance(ICalendar.class, c.uid);
		ElasticsearchClient esClient = ESearchActivator.getClient();
		VEventIndexStore indexStore = new VEventIndexStore(esClient, c, DataSourceRouter.location(ctx, c.uid));
		return new CalendarIndexer(ctx.provider().instance(ICalendar.class, c.uid), calendarApi,
				new VEventSeriesIndexWriter(indexStore));
	}

	public static class VEventSeriesIndexWriter implements IndexWriter<VEventSeries> {

		private VEventIndexStore calendarIndexStore;

		public VEventSeriesIndexWriter(VEventIndexStore cidx) {
			this.calendarIndexStore = cidx;
		}

		@Override
		public void write(ItemValue<VEventSeries> toIndex) {
			calendarIndexStore.update(toIndex.item(), toIndex.value);

		}

		@Override
		public void deleteByIds(List<Long> itemIds) {
			itemIds.forEach(item -> calendarIndexStore.delete(item));
		}
	}

	public static class CalendarIndexer extends AbstractTypeIndexer<VEventSeries> {

		public CalendarIndexer(ICalendar instance, ICrudFactory<VEventSeries> calendarApi,
				IndexWriter<VEventSeries> writer) {
			super(instance, calendarApi, writer);
		}

	}

}
