/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.endpoint.parsing;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayDeque;
import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Handler;
import net.bluemind.imap.endpoint.EndpointRuntimeException;
import net.bluemind.imap.endpoint.cmd.RawImapCommand;
import net.bluemind.imap.endpoint.parsing.Part.Type;
import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.memory.pool.api.ChunkFiler.NamedChunk;

public class ImapRequestParser {

	private static final Path TMP = Paths.get(System.getProperty("java.io.tmpdir"));

	private static final Logger logger = LoggerFactory.getLogger(ImapRequestParser.class);

	private final ArrayDeque<Part> parts;
	private final Handler<RawImapCommand> cmdHandler;

	private Part prev;

	public ImapRequestParser(Handler<RawImapCommand> parsedCmdHandler) {
		this.cmdHandler = parsedCmdHandler;
		parts = new ArrayDeque<>();
		if (logger.isDebugEnabled()) {
			logger.debug("parser created with handler {}", parsedCmdHandler);
		}
	}

	public void parse(Part p) {
		Part current = p;

		// merge literals as single part
		if (p.type() == Type.LITERAL_CHUNK) {
			// first chunk of literal
			if (prev.type() == Type.COMMAND) {
				int fullSize = current.buffer().readableBytes() + p.expected();
				NamedChunk nc = chunk(fullSize);
				nc.chunk().append(current.buffer());
				current = Part.literalChunk(nc, null, p.expected());
				parts.add(current);
				prev = current;
			} else {
				prev.writeBytes(current.buffer());
			}
		} else {
			parts.add(current);
			prev = current;
			if (!current.continued()) {
				// parts represent a full command
				ArrayList<Part> copy = new ArrayList<>(parts);
				parts.clear();
				RawImapCommand cmd = new RawImapCommand(copy);
				cmdHandler.handle(cmd);
			}
		}

	}

	private NamedChunk chunk(int size) {
		try {
			return ChunkFiler.newNamedChunk(size);
		} catch (IOException e) {
			throw new EndpointRuntimeException(e);
		}
	}

	public void close() {
		parts.removeIf(p -> {
			p.release();
			return true;
		});

	}

}
