/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.endpoint.exec;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.StampedLock;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;

import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import net.bluemind.imap.endpoint.ImapContext;
import net.bluemind.imap.endpoint.SessionState;
import net.bluemind.imap.endpoint.cmd.IdleCommand;
import net.bluemind.imap.endpoint.driver.ImapIdSet;
import net.bluemind.imap.endpoint.driver.MailPart;
import net.bluemind.imap.endpoint.driver.MailPartBuilder;
import net.bluemind.imap.endpoint.driver.MailboxConnection;
import net.bluemind.imap.endpoint.driver.MailboxConnection.IdleConsumer;
import net.bluemind.imap.endpoint.driver.SelectedFolder;
import net.bluemind.imap.endpoint.driver.SelectedMessage;
import net.bluemind.imap.endpoint.locks.ISequenceCheckpoint;
import net.bluemind.imap.endpoint.locks.MailboxSequenceLocks;
import net.bluemind.imap.endpoint.locks.MailboxSequenceLocks.MailboxSeqLock;
import net.bluemind.imap.endpoint.locks.MailboxSequenceLocks.OpCompletionListener;
import net.bluemind.lib.vertx.Result;

public class IdleProcessor extends AuthenticatedCommandProcessor<IdleCommand> {

	private static final Logger logger = LoggerFactory.getLogger(IdleProcessor.class);

	private final ISequenceCheckpoint idleCheckpointer = new ISequenceCheckpoint() {
	};
	private final List<MailPart> parts = List.of(MailPartBuilder.named("FLAGS"), MailPartBuilder.named("UID"));

	@Override
	public void checkedOperation(IdleCommand command, ImapContext ctx, Handler<AsyncResult<Void>> completed) {
		ctx.idlingTag(command.raw().tag());
		ctx.state(SessionState.IDLING);
		MailboxConnection mailbox = ctx.mailbox();
		try {
			logger.info("Monitoring {}", ctx.selected());
			ctx.write("+ idling\r\n");
			Lock exclusiveLock = new StampedLock().asReadWriteLock().writeLock();
			onChange(command, ctx, mailbox, new SelectedMessage[0], exclusiveLock).whenComplete((v, ex) -> {
				if (ex != null) {
					ctx.write(command.raw().tag() + " NO unknown error: " + ex.getMessage() + "\r\n");
					completed.handle(Result.fail(ex));
				} else {
					IdleConsumer cons = new IdleConsumer() {
						private AtomicBoolean killed = new AtomicBoolean();

						@Override
						public void accept(SelectedMessage[] changed) {
							if (!killed.get()) {
								onChange(command, ctx, mailbox, changed, exclusiveLock);
							}
						}

						@Override
						public CompletableFuture<Void> shutdown() {
							killed.set(true);
							return waitForCompletion(command, ctx, exclusiveLock);
						}

					};
					mailbox.idleMonitor(ctx.selected(), cons);
					completed.handle(Result.success());
				}
			});
		} catch (Exception e) {
			ctx.write(command.raw().tag() + " NO unknown error: " + e.getMessage() + "\r\n");
			completed.handle(Result.fail(e));
		}
	}

	private CompletableFuture<Void> waitForCompletion(IdleCommand command, ImapContext ctx, Lock exclusiveLock) {
		CompletableFuture<Void> comp = new CompletableFuture<>();
		Thread.ofVirtual().name("kill-idle-" + command.raw().tag()).start(() -> {
			var chrono = Stopwatch.createStarted();
			exclusiveLock.lock();
			ctx.vertxContext.executeBlocking(() -> {
				logger.info("Shutdown of {} after {}ms", command.raw().tag(), chrono.elapsed(TimeUnit.MILLISECONDS));
				return comp.complete(null);
			});
		});
		return comp;
	}

	private CompletableFuture<Void> onChange(IdleCommand command, ImapContext ctx, MailboxConnection mailbox,
			SelectedMessage[] changed, Lock exclusiveLock) {

		if (!exclusiveLock.tryLock()) {// NOSONAR if this lock is not unlocked we would just loose notifications from
										// the current idle channel only
			return CompletableFuture.completedFuture(null);
		}
		MailboxSeqLock lockSupport = MailboxSequenceLocks.forMailbox(ctx.selected());
		return lockSupport.withReadLock(ctx, "Idle{" + command.raw().tag() + "}").thenCompose(lockedOp -> {
			try {
				return lockedCheckpoint(command, ctx, mailbox, changed, lockedOp);
			} catch (Exception e) {
				logger.error("Error with idle readlock", e);
				lockedOp.complete();
				return CompletableFuture.failedFuture(e);
			}
		}).whenComplete((v, ex) -> exclusiveLock.unlock());
	}

	private CompletableFuture<Void> lockedCheckpoint(IdleCommand command, ImapContext ctx, MailboxConnection mailbox,
			SelectedMessage[] changed, OpCompletionListener lockedOp) {
		CompletableFuture<Void> cpComplete = new CompletableFuture<>();
		Callable<Promise<Void>> checkpointer = () -> {
			Promise<Void> prom = Promise.promise();
			try {
				idleCheckpoint(command, ctx, mailbox, changed, prom);
			} catch (Exception e) {
				prom.fail(e);
			}
			return prom;
		};

		ctx.vertxContext.executeBlocking(checkpointer, true).andThen(ar -> {
			if (ar.failed()) {
				lockedOp.complete();
				ctx.vertxContext.runOnContext(v -> cpComplete.completeExceptionally(ar.cause()));
			} else {
				ar.result().future().andThen(cp -> {
					lockedOp.complete();
					if (cp.failed()) {
						ctx.vertxContext.runOnContext(v -> cpComplete.completeExceptionally(cp.cause()));
					} else {
						ctx.vertxContext.runOnContext(v -> cpComplete.complete(null));
					}
				});
			}
		});

		return cpComplete;
	}

	private void idleCheckpoint(IdleCommand command, ImapContext ctx, MailboxConnection mailbox,
			SelectedMessage[] changed, Promise<Void> prom) {
		StringBuilder sb = new StringBuilder();
		idleCheckpointer.checkpointSequences(logger, "idle", sb, ctx);
		SelectedFolder live = ctx.selected();
		ctx.write(sb.toString());
		logger.trace("idle checkpoint for {}", live);
		if (changed.length > 0) {
			FetchedItemStream fetchStream = new FetchedItemStream(ctx, command.raw().tag() + " idle", parts, true);
			ImapIdSet changeSet = ImapIdSet.uids(
					Arrays.stream(changed).map(sm -> Long.toString(sm.imapUid())).collect(Collectors.joining(",")));
			mailbox.fetch(live, changeSet, parts, fetchStream).whenComplete((done, ex) -> {
				if (ex != null) {
					prom.fail(ex);
				} else {
					prom.complete();
				}
			});
		} else {
			prom.complete();
		}
	}

	@Override
	public Class<IdleCommand> handledType() {
		return IdleCommand.class;
	}

}
