/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.imap.endpoint.exec;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;
import net.bluemind.imap.endpoint.EndpointRuntimeException;
import net.bluemind.imap.endpoint.ImapContext;
import net.bluemind.imap.endpoint.SessionState;
import net.bluemind.imap.endpoint.cmd.AbstractFolderNameCommand;
import net.bluemind.imap.endpoint.cmd.AnalyzedCommand;
import net.bluemind.imap.endpoint.driver.MailboxConnection;
import net.bluemind.imap.endpoint.driver.SelectedFolder;
import net.bluemind.imap.endpoint.locks.ISequenceReader;

public abstract class AbstractSelectorProcessor<T extends AbstractFolderNameCommand>
		extends AuthenticatedCommandProcessor<T> implements ISequenceReader {
	private static final Logger logger = LoggerFactory.getLogger(AbstractSelectorProcessor.class);

	protected boolean isAlwaysReadOnly() {
		return true;
	}

	@Override
	public void checkedOperation(T sc, ImapContext ctx, Handler<AsyncResult<Void>> completed) {
		MailboxConnection con = ctx.mailbox();
		long time = System.currentTimeMillis();
		SelectedFolder selected = null;
		try {
			selected = con.select(sc.folder());
		} catch (EndpointRuntimeException e) {
			selected = null;
		}

		if (selected == null) {
			missingFolder(sc, ctx, completed);
			return;
		}

		StringBuilder resp = new StringBuilder();
		resp.append("* " + selected.exist + " EXISTS\r\n");
		resp.append("* 0 RECENT\r\n");
		resp.append("* FLAGS (\\Answered \\Flagged \\Draft \\Deleted \\Seen" + extraLabels(selected) + ")\r\n");
		resp.append("* OK [PERMANENTFLAGS (\\Answered \\Flagged \\Draft \\Deleted \\Seen \\*)] Ok\r\n");
		resp.append("* OK [UNSEEN " + selected.unseen + "] Ok\r\n");
		resp.append("* OK [UIDVALIDITY " + selected.folder.value.uidValidity + "] Ok\r\n");
		resp.append("* OK [UIDNEXT " + (selected.folder.value.lastUid + 1) + "] Ok\r\n");
		if (isAlwaysReadOnly() || selected.mailbox.readOnly) {
			resp.append(sc.raw().tag() + " OK [READ-ONLY] Completed\r\n");
			selected.mailbox.readOnly = true;
		} else {
			resp.append(sc.raw().tag() + " OK [READ-WRITE] Completed\r\n");
		}

		ctx.state(SessionState.SELECTED);
		ctx.selected(selected);
		ctx.write(resp.toString()).onComplete(completed);
		time = System.currentTimeMillis() - time;
		if (logger.isTraceEnabled()) {
			logger.trace("Selected in {}ms {} => {} ", time, sc.folder(), selected.folder);
		}
	}

	private String extraLabels(SelectedFolder selected) {
		String labels = selected.labels.stream().collect(Collectors.joining(" "));
		if (!labels.isBlank()) {
			return " " + labels;
		} else {
			return "";
		}
	}

	@Override
	public CompletableFuture<SelectedFolder> readFolder(AnalyzedCommand cmd, ImapContext ctx) {
		return ctx.vertxContext.executeBlocking(//
				() -> ctx.mailbox().select(((AbstractFolderNameCommand) cmd).folder())//
		).toCompletionStage().toCompletableFuture();
	}

	private void missingFolder(T sc, ImapContext ctx, Handler<AsyncResult<Void>> completed) {

		if (ctx.state() == SessionState.SELECTED) {
			ctx.state(SessionState.AUTHENTICATED);
			ctx.selected(null);
			ctx.write("* OK [CLOSED] Ok\r\n");
		}
		ctx.write(sc.raw().tag() + " NO Mailbox does not exist\r\n").onComplete(completed);
	}
}
