/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.imap.endpoint.locks;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.StampedLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.Stopwatch;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Timer;

import io.vertx.core.Context;
import net.bluemind.imap.endpoint.ImapContext;
import net.bluemind.imap.endpoint.driver.SelectedFolder;
import net.bluemind.lifecycle.helper.SoftReset;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public class MailboxSequenceLocks {
	public interface OpCompletionListener {

		public static final OpCompletionListener NOOP = () -> {
		};

		void complete();
	}

	public interface MailboxSeqLock {
		public static final MailboxSeqLock NOOP = new MailboxSeqLock() {

			@Override
			public CompletableFuture<OpCompletionListener> withWriteLock(ImapContext ctx, Object grabber) {
				return CompletableFuture.completedFuture(OpCompletionListener.NOOP);
			}

			@Override
			public CompletableFuture<OpCompletionListener> withReadLock(ImapContext ctx, Object grabber) {
				return CompletableFuture.completedFuture(OpCompletionListener.NOOP);
			}
		};

		CompletableFuture<OpCompletionListener> withReadLock(ImapContext ctx, Object grabber);

		CompletableFuture<OpCompletionListener> withWriteLock(ImapContext ctx, Object grabber);
	}

	private static final Cache<String, MailboxSeqLock> locks = initLocksCache();

	private static final Cache<String, MailboxSeqLock> initLocksCache() {
		Cache<String, MailboxSeqLock> cache = Caffeine.newBuilder().expireAfterAccess(20, TimeUnit.MINUTES)
				.recordStats().build();
		SoftReset.register(cache::invalidateAll);
		return cache;
	}

	private static class MetricsSupport {
		final Counter lockFailures;
		final Timer lockDurations;

		public MetricsSupport() {
			Registry reg = MetricsRegistry.get();
			IdFactory idf = new IdFactory("imap-locks", reg, MetricsSupport.class);
			lockFailures = reg.counter(idf.name("grab-failures"));
			lockDurations = reg.timer(idf.name("durations"));
		}
	}

	private static final MetricsSupport metrics = new MetricsSupport();

	public static MailboxSeqLock forMailbox(SelectedFolder sel) {
		if (sel == null) {
			return MailboxSeqLock.NOOP;
		}
		return forMailbox(sel.mailbox.owner.uid);
	}

	public static MailboxSeqLock forMailbox(String owner) {
		return locks.get(owner, SeqLockImpl::new);

	}

	private static final class SeqLockImpl implements MailboxSeqLock {

		private static final Logger logger = LoggerFactory.getLogger(SeqLockImpl.class);
		private final ReadWriteLock lock;
		private final String owner;

		public SeqLockImpl(String owner) {
			this.owner = owner;
			this.lock = new StampedLock().asReadWriteLock();
		}

		@Override
		public CompletableFuture<OpCompletionListener> withWriteLock(ImapContext ctx, Object grabber) {
			return withLock(lock.writeLock(), ctx, grabber);
		}

		@Override
		public CompletableFuture<OpCompletionListener> withReadLock(ImapContext ctx, Object grabber) {
			return withLock(lock.readLock(), ctx, grabber);
		}

		private CompletableFuture<OpCompletionListener> withLock(Lock l, ImapContext ctx, Object grabber) {
			CompletableFuture<OpCompletionListener> ret = new CompletableFuture<>();
			lockInContext(l, ctx, ret, grabber, Stopwatch.createStarted());
			return ret;
		}

		private void lockInContext(Lock l, ImapContext imapContext, CompletableFuture<OpCompletionListener> ret,
				Object grabber, Stopwatch since) {
			Instant grab = Instant.now();
			Context vertxContext = imapContext.vertxContext;

			CompletableFuture<Void> vertxError = new CompletableFuture<>();
			imapContext.childExceptionHandler(t -> {
				logger.error("{} Error in the locking process of {} ({})", imapContext, grabber, t.getClass());
				vertxError.completeExceptionally(t);
			});

			Thread.ofVirtual().name("grab-lock-for:" + grabber).start(() -> {
				boolean grabbed = false;
				try {
					grabbed = l.tryLock(60, TimeUnit.SECONDS);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					vertxContext.executeBlocking(() -> ret.completeExceptionally(e));
					return;
				}
				if (!grabbed) {
					metrics.lockFailures.increment();
					String msg = "[%s] Failed to grab lock for %s in 60s".formatted(owner, grabber);
					vertxContext.executeBlocking(() -> ret.completeExceptionally(new LockingRuntimeException(msg)));
					return;
				}
				long ms = since.elapsed(TimeUnit.MILLISECONDS);
				if (ms > 100 && logger.isWarnEnabled()) {
					logger.warn("[{}] Took {}ms to acquire {} for {}", owner, ms, l, grabber);
				}

				Stopwatch lockDuration = Stopwatch.createStarted();
				Thread monVt = startMonitor(grabber, vertxContext, grab);
				Runnable unlockOnce = onlyOnce(() -> {
					l.unlock();
					monVt.interrupt();
				}, grabber);
				imapContext.runOnShutdown(unlockOnce);
				if (vertxError.isCompletedExceptionally()) {
					unlockOnce.run();
					vertxContext.executeBlocking(
							() -> ret.completeExceptionally(new LockingRuntimeException("Lock cancelled")));
					return;
				}
				imapContext.childExceptionHandler(null);
				OpCompletionListener whenDone = () -> {
					unlockOnce.run();
					warnOnLongHold(l, grabber, lockDuration);
				};
				vertxContext.executeBlocking(() -> ret.complete(whenDone));
			});
		}

		private Runnable onlyOnce(Runnable r, Object grabber) {
			AtomicBoolean ran = new AtomicBoolean();
			return new Runnable() {
				@Override
				public void run() {
					if (ran.compareAndSet(false, true)) {
						r.run();
					}
				}

				@Override
				public String toString() {
					return super.toString() + " for " + grabber;
				}
			};
		}

		private void warnOnLongHold(Lock l, Object grabber, Stopwatch lockDuration) {
			long lockedMs = lockDuration.elapsed(TimeUnit.MILLISECONDS);
			metrics.lockDurations.record(lockedMs, TimeUnit.MILLISECONDS);
			if (lockedMs > 200) {
				logger.warn("[{}] SLOW Release lock {} for {}, lock was held {}ms", owner, l, grabber, lockedMs);
			} else {
				logger.trace("[{}] Release lock {} for {}", owner, l, grabber);
			}
		}

		private Thread startMonitor(Object grabber, Context vertxContext, Instant grab) {
			return Thread.ofVirtual().name("grab-monitor-for:" + grabber).start(() -> {
				while (true) {
					try {
						Thread.sleep(20000);
						vertxContext.executeBlocking(() -> {
							logger.warn("[{}] still holding lock for {} since {}", owner, grabber, grab);
							return null;
						});
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
						break;
					}
				}
			});
		}
	}

}
