/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.endpoint.events;

import java.util.LinkedList;
import java.util.List;

import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.EventBus;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageCodec;
import io.vertx.core.eventbus.MessageConsumer;
import net.bluemind.imap.endpoint.SessionState;
import net.bluemind.imap.endpoint.driver.MailboxConnection;

public class EventNexus {

	private final EventBus eb;
	private List<MessageConsumer<?>> resources;
	private String id;
	private final DeliveryOptions delOps;

	public EventNexus(String id, EventBus eb) {
		this.id = id;
		this.eb = eb;
		resources = new LinkedList<>();
		MessageCodec<SeqChange, SeqChange> seqChangeRecordCodec = new MessageCodec<SeqChange, SeqChange>() {

			@Override
			public void encodeToWire(Buffer buffer, SeqChange s) {
				// not clustered
			}

			@Override
			public SeqChange decodeFromWire(int pos, Buffer buffer) {
				// not clustered
				return null;
			}

			@Override
			public SeqChange transform(SeqChange s) {
				return s;
			}

			@Override
			public String name() {
				return "seqChange";
			}

			@Override
			public byte systemCodecID() {
				return -1;
			}
		};
		try {
			eb.registerCodec(seqChangeRecordCodec);
		} catch (IllegalStateException ise) {
			// already registered
		}
		this.delOps = new DeliveryOptions().setLocalOnly(true).setCodecName("seqChange");
	}

	public void close() {
		resources.removeIf(cons -> {
			cons.unregister();
			return true;
		});
	}

	public void addStateListener(StateChangeListener scl) {
		MessageConsumer<String> cons = eb.consumer(id + ".imap.ep.state", (Message<String> state) -> {
			SessionState parsedState = SessionState.valueOf(state.body());
			scl.stateChanged(parsedState);
		});
		resources.add(cons);
	}

	public void dispatchStateChanged(SessionState authenticated) {
		eb.publish(id + ".imap.ep.state", authenticated.name());
	}

	public static record SeqChange(String tag, String conId, String folderUid, long version) {

	}

	public void addSequenceChangeListener(MailboxConnection mc, SequencesChangeListener scl) {
		MessageConsumer<SeqChange> cons = eb.localConsumer(mc.login() + ".imap.ep.size", (Message<SeqChange> state) -> {
			SeqChange body = state.body();
			// ignore notifications from self
			if (!id.equals(body.conId())) {
				scl.markDirty(body.tag(), body.folderUid(), body.version());
			}
		});
		resources.add(cons);
	}

	public void dispatchSequencesChanged(MailboxConnection mc, String fromTag, String folderUid, long newVersion) {
		SeqChange change = new SeqChange(fromTag, id, folderUid, newVersion);
		eb.publish(mc.login() + ".imap.ep.size", change, delOps);
	}

}
