/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.driver.mailapi;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import org.apache.james.mime4j.dom.Body;
import org.apache.james.mime4j.dom.Entity;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.Multipart;
import org.apache.james.mime4j.dom.SingleBody;
import org.apache.james.mime4j.message.MessageImpl;
import org.apache.james.mime4j.message.MultipartImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.CharMatcher;
import com.google.common.base.Strings;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.streams.ReadStream;
import net.bluemind.backend.mail.api.DispositionType;
import net.bluemind.backend.mail.api.MessageBody.Part;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.WithId;
import net.bluemind.core.api.Stream;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.lib.vertx.utils.MmapWriteStream;
import net.bluemind.mime4j.common.Mime4JHelper;

public class BodyStructureRenderer {
	private static final Logger logger = LoggerFactory.getLogger(BodyStructureRenderer.class);
	private static final ByteBuf TWO_CRLF = Unpooled.wrappedBuffer("\r\n\r\n".getBytes());

	public String from(IDbMailboxRecords recApi, WithId<MailboxRecord> rec, Part root, int bodySize) {
		StringBuilder sb = new StringBuilder();
		List<String> addresses = new ArrayList<>();
		getPartsAddresses(root, addresses);

		Map<String, Integer> undecodedPartSizes = getEncodedPartSizes(recApi, rec, addresses, bodySize);
		from0(sb, root, undecodedPartSizes);
		return sb.toString();
	}

	private Map<String, Integer> getEncodedPartSizes(IDbMailboxRecords recApi, WithId<MailboxRecord> rec,
			List<String> addresses, int bodySize) {
		Stream fullMsg = recApi.fetchComplete(rec.value.imapUid);
		Map<String, Integer> sizes = new HashMap<>();
		ByteBuf emlBuffer = readMmap(fullMsg, bodySize * 2).join();
		try (InputStream in = new ByteBufInputStream(emlBuffer, true); Message parsed = Mime4JHelper.parse(in, false)) {
			Body bd = null;
			for (String address : addresses) {
				if (parsed.isMultipart()) {
					Multipart mp = (Multipart) parsed.getBody();
					bd = Mime4JHelper.expandTree(mp.getBodyParts()).stream()
							.filter(ae -> address.equals(ae.getMimeAddress())).findAny().map(Entity::getBody)
							.orElseGet(() -> {
								logger.warn("Part {} not found", address);
								return null;
							});
				} else if (address.equals("1") || address.equals("TEXT")) {
					bd = parsed.getBody();
				}
				if (bd == null) {
					sizes.put(address, 0);
				} else {
					sizes.put(address, buffer(bd).readableBytes());
				}
			}
		} catch (Exception e) {
			logger.error("cannot retrieve part sizes:{}", e);
			return Collections.emptyMap();
		}
		return sizes;
	}

	private void getPartsAddresses(Part root, List<String> addresses) {
		String[] mimeSplit = root.mime.split("/");
		if (mimeSplit[0].equalsIgnoreCase("multipart")) {
			for (Part c : root.children) {
				getPartsAddresses(c, addresses);
			}
		} else {
			addresses.add(root.address);
		}
	}

	private void from0(StringBuilder sb, Part root, Map<String, Integer> encodedPartSizes) {
		String[] mimeSplit = root.mime.split("/");
		if (mimeSplit[0].equalsIgnoreCase("multipart")) {
			sb.append("(");
			for (Part c : root.children) {
				from0(sb, c, encodedPartSizes);
			}
			sb.append(" \"").append(mimeSplit[1].toUpperCase());
			String boundary = (root.mimeParameters != null && root.mimeParameters.containsKey("boundary"))
					? root.mimeParameters.get("boundary")
					: "-=Part." + root.address + "=-";
			sb.append("\" (\"BOUNDARY\" \"").append(boundary).append("\") NIL NIL NIL");
			sb.append(")");
		} else {
			sb.append("(\"").append(mimeSplit[0].toUpperCase()).append("\" \"").append(mimeSplit[1].toUpperCase())
					.append("\"");
			// cs
			if (mimeSplit[0].equalsIgnoreCase("text")) {
				sb.append(" (\"CHARSET\" \"" + Optional.ofNullable(root.charset).orElse("us-ascii") + "\")");
			} else {
				if (!Strings.isNullOrEmpty(root.fileName)) {
					sb.append(" (\"NAME\" \"").append(root.fileName).append("\")");
				} else {
					sb.append(" NIL");
				}
			}
			// contentId
			if (root.contentId != null) {
				sb.append(" \"").append(root.contentId).append("\"");
			} else {
				sb.append(" NIL");
			}

			// nil
			sb.append(" NIL");

			// encoding
			sb.append(" \"").append(root.encoding.toUpperCase()).append("\"");

			// size
			int size = encodedPartSizes.computeIfAbsent(root.address, v -> root.size);
			sb.append(" ").append(size);

			if (root.dispositionType == DispositionType.ATTACHMENT && !Strings.isNullOrEmpty(root.fileName)) {
				if (mimeSplit[0].equalsIgnoreCase("text")) {
					sb.append(" 0 NIL (\"ATTACHMENT\" (\"FILENAME\" \"" + cleanFileName(root) + "\")) NIL NIL");
				} else {
					sb.append(" NIL (\"ATTACHMENT\" (\"FILENAME\" \"" + cleanFileName(root) + "\" \"SIZE\" \"" + size
							+ "\")) NIL NIL");
				}
			} else if (root.dispositionType == DispositionType.INLINE && root.contentId != null) {
				// RFC 3501 : "A body type of type TEXT contains, immediately after the basic
				// fields, the size of the body in text lines."
				if (mimeSplit[0].equalsIgnoreCase("text")) {
					sb.append(" 0 NIL (\"INLINE\" NIL) NIL NIL");
				} else {
					sb.append(" NIL (\"INLINE\" ");
					if (!Strings.isNullOrEmpty(root.fileName)) {
						sb.append("(\"FILENAME\" \"").append(root.fileName).append("\")");
					} else {
						sb.append("NIL");
					}
					sb.append(") NIL NIL");
				}
			} else {
				sb.append(" 1 NIL NIL NIL NIL");
			}

			sb.append(")");
		}
	}

	private String cleanFileName(Part p) {
		String s = Normalizer.normalize(p.fileName, Normalizer.Form.NFD);
		s = s.replaceAll("[\\p{InCombiningDiacriticalMarks}]", "");
		return CharMatcher.ascii().retainFrom(s);
	}

	private static final Path TMP = Paths.get(System.getProperty("java.io.tmpdir"));

	private CompletableFuture<ByteBuf> readMmap(Stream s, int sizeHint) {
		try {
			MmapWriteStream out = new MmapWriteStream(TMP, sizeHint);
			ReadStream<Buffer> toRead = VertxStream.read(s);
			toRead.pipeTo(out);
			toRead.resume();
			return out.mmap();
		} catch (IOException e) {
			return CompletableFuture.failedFuture(e);
		}
	}

	private ByteBuf buffer(Body b) throws IOException {
		if (b instanceof SingleBody body) {
			try (InputStream in = body.getInputStream()) {
				return Unpooled.wrappedBuffer(in.readAllBytes());
			}
		} else if (b instanceof MultipartImpl mp) {
			try (Message m = new MessageImpl()) {
				mp.getParent().getHeader().getFields().forEach(f -> m.getHeader().addField(f));
				m.setBody(mp);
				ByteBuf out = Unpooled.buffer();
				Mime4JHelper.serialize(m, new ByteBufOutputStream(out));
				int endOfHeader = ByteBufUtil.indexOf(TWO_CRLF.duplicate(), out);
				if (endOfHeader > 0) {
					out = out.skipBytes(endOfHeader + 4);
				}
				return out;
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
				return Unpooled.buffer();
			}
		} else {
			return Unpooled.buffer();
		}

	}
}
