/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.imap.driver.mailapi.search;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.backend.mail.api.Conversation;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.RawImapBinding;
import net.bluemind.backend.mail.replica.api.WithId;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.imap.endpoint.cmd.ImapDateParser;
import net.bluemind.imap.endpoint.driver.SelectedFolder;

public class UidSearchFastPaths {

	public interface SearchFastPath {
		List<Long> search(int mgetSize, SelectedFolder sel, String query);

		default boolean supports(String query) {
			return false;
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(UidSearchFastPaths.class);
	private static final List<SearchFastPath> tunedSearches = List.of(new OutlookWithRangeSince(),
			new OutlookUidSince(), new OutlookSince(), new OutlookSinceAll(), new OutlookNotDeleted(), new UidAfter(),
			new IPhoneDeleted(), new IPhoneUnseen(), new VisibleUnread(), new RoundCubeDrafts(),
			new VisibleUnreadWithRange(), new ImapSyncAll(), new UnseenUndeleted());

	public static Optional<SearchFastPath> lookup(String query) {
		return tunedSearches.stream().filter(sp -> sp.supports(query)).findAny().map(fp -> {
			logger.info("Using fast-path {} for '{}'", fp, query);
			return fp;
		});
	}

	private static class OutlookSince implements SearchFastPath {

		private static final Pattern outlookSearch = Pattern.compile("since ([^\s]+)$", Pattern.CASE_INSENSITIVE);
		private OutlookUidSince delegate = new OutlookUidSince();

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			return delegate.search(mgetSize, sel, "uid 1:* " + query);
		}

		@Override
		public boolean supports(String query) {
			return outlookSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "OutlookSince";
		}

	}

	private static class OutlookWithRangeSince implements SearchFastPath {
		private static final Pattern outlookWithRangeSearch = Pattern.compile("1:\\* since ([^\s]+)$",
				Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			var matcher = outlookWithRangeSearch.matcher(query);
			matcher.find();
			Date date = ImapDateParser.readDate(matcher.group(1));

			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "");
			List<Long> items = raw.stream().map(r -> r.itemId).toList();
			List<Long> output = new ArrayList<>(raw.size());
			for (var slice : Lists.partition(items, mgetSize)) {
				List<WithId<MailboxRecord>> loaded = sel.recApi.lightSlice(slice);
				for (var it : loaded) {
					if (date.before(it.value.internalDate)) {
						output.add(it.value.imapUid);
					}
				}
			}
			return output;
		}

		@Override
		public boolean supports(String query) {
			return outlookWithRangeSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "OutlookWithRangeSince";
		}

	}

	private static class OutlookSinceAll implements SearchFastPath {

		private static final Pattern outlookSearch = Pattern.compile("since ([^\s]+) all$", Pattern.CASE_INSENSITIVE);
		private OutlookUidSince delegate = new OutlookUidSince();

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			var m = outlookSearch.matcher(query);
			m.find();
			String date = m.group(1);
			return delegate.search(mgetSize, sel, "uid 1:* since " + date);
		}

		@Override
		public boolean supports(String query) {
			return outlookSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "OutlookSinceAll";
		}

	}

	private static class OutlookUidSince implements SearchFastPath {

		private static final Pattern outlookSearch = Pattern.compile("uid ([^\s]+) since ([^\s]+)$",
				Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			var matcher = outlookSearch.matcher(query);
			matcher.find();
			String uidSet = matcher.group(1);
			Date date = ImapDateParser.readDate(matcher.group(2));

			List<RawImapBinding> raw = sel.recApi.imapIdSet(uidSet, "");
			List<Long> items = raw.stream().map(r -> r.itemId).toList();
			List<Long> output = new ArrayList<>(raw.size());
			for (var slice : Lists.partition(items, mgetSize)) {
				List<WithId<MailboxRecord>> loaded = sel.recApi.lightSlice(slice);
				for (var it : loaded) {
					if (date.before(it.value.internalDate)) {
						output.add(it.value.imapUid);
					}
				}
			}

			return output;
		}

		@Override
		public boolean supports(String query) {
			return outlookSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "OutlookUidSince";
		}

	}

	private static class UidAfter implements SearchFastPath {
		private static final Pattern appleMail = Pattern.compile("uid ([^:]+):([^\s]+)$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			var matcher = appleMail.matcher(query);
			matcher.find();
			List<RawImapBinding> raw = sel.recApi.imapIdSet(matcher.group(1) + ":" + matcher.group(2), "");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return appleMail.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "UidAfter";
		}

	}

	private static class OutlookNotDeleted implements SearchFastPath {

		private static final Pattern outlookSearch = Pattern.compile("not deleted$", Pattern.CASE_INSENSITIVE);
		private static final Pattern outlookAltSearch = Pattern.compile("1:\\* not deleted$", Pattern.CASE_INSENSITIVE);
		private static final Pattern notOutlookSearch = Pattern.compile("all undeleted$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "-deleted");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return outlookSearch.matcher(query).matches() //
					|| outlookAltSearch.matcher(query).matches() //
					|| notOutlookSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "NotDeleted";
		}

	}

	private static class ImapSyncAll implements SearchFastPath {

		private static final Pattern allSearch = Pattern.compile("all$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return allSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "ImapSyncAll";
		}

	}

	private static class VisibleUnread implements SearchFastPath {

		private static final Pattern delSearch = Pattern.compile("all undeleted unseen$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "-deleted,-seen");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return delSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "VisibleUnread";
		}

	}

	private static class UnseenUndeleted implements SearchFastPath {

		private static final Pattern search = Pattern.compile("^unseen undeleted$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "-deleted,-seen");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return search.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "UnseenUndeleted";
		}

	}

	private static class VisibleUnreadWithRange implements SearchFastPath {

		private static final Pattern delSearchWithRange = Pattern.compile("uid ([^\s]+):([^\s]+) unseen undeleted$",
				Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {

			var matcher = delSearchWithRange.matcher(query);
			matcher.find();
			List<RawImapBinding> raw = sel.recApi.imapIdSet(matcher.group(1) + ":" + matcher.group(2),
					"-deleted,-seen");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return delSearchWithRange.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "VisibleUnreadWithRange";
		}

	}

	private static class IPhoneDeleted implements SearchFastPath {

		private static final Pattern delSearch = Pattern.compile("1:\\* deleted$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "+deleted");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return delSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "Deleted";
		}

	}

	private static class IPhoneUnseen implements SearchFastPath {

		private static final Pattern unseenSearch = Pattern.compile("1:\\* unseen$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			List<RawImapBinding> raw = sel.recApi.imapIdSet("1:*", "-seen");
			return raw.stream().map(r -> r.imapUid).toList();
		}

		@Override
		public boolean supports(String query) {
			return unseenSearch.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "Unseen";
		}

	}

	private static class RoundCubeDrafts implements SearchFastPath {
		private static final Pattern rcDrafts = Pattern.compile("^HEADER Message-ID \\S+$", Pattern.CASE_INSENSITIVE);

		@Override
		public List<Long> search(int mgetSize, SelectedFolder sel, String query) {
			int last = query.lastIndexOf(' ');

			String lastArg = query.substring(last).trim();
			final String messageId = lastArg.startsWith("<") && lastArg.endsWith(">") ? lastArg : "<" + lastArg + ">";
			Long lookedUp = sel.convRefApi.lookup(messageId, Collections.emptySet());
			try {
				Conversation conversation = sel.convApi.get(Long.toHexString(lookedUp));
				List<WithId<MailboxRecord>> slice = sel.recApi
						.slice(conversation.messageRefs.stream().map(m -> m.itemId).toList());
				MessageBody body = sel.sourceBodies.multiple(slice.stream().map(s -> s.value.messageBody).toList()) //
						.stream().filter(b -> b.messageId.equals(messageId)) //
						.findFirst().orElse(null);
				return body == null ? Collections.emptyList() //
						: slice.stream().filter(m -> m.value.messageBody.equals(body.guid)).map(s -> s.value.imapUid)
								.toList();
			} catch (ServerFault e) {
				logger.error(e.getMessage());
				return Collections.emptyList();
			}
		}

		@Override
		public boolean supports(String query) {
			return rcDrafts.matcher(query).matches();
		}

		@Override
		public String toString() {
			return "RoundCubeDrafts";
		}

	}

}
