/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.group.service.internal.cql;

import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.utils.ValidationResult;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.cql.common.CqlContStoreService;
import net.bluemind.directory.repository.DirEntryNG;
import net.bluemind.directory.repository.IDirEntryStore.ItemIdAndRole;
import net.bluemind.domain.api.Domain;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.GroupSearchQuery;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.group.service.GroupHelper;
import net.bluemind.group.service.IInCoreGroup;

public class CqlGroupService implements IGroup, IInCoreGroup {

	private final CqlContStoreService store;
	private static final DirGroupAdapter adapter = new DirGroupAdapter();

	public CqlGroupService(BmContext ctx, ItemValue<Domain> domain, Container container) {
		this.store = new CqlContStoreService(ctx, container, domain);

	}

	@Override
	public void setExtId(String uid, String extId) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public List<ItemValue<Group>> memberOf(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<String> memberOfGroups(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public ValidationResult validate(String[] uids) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void restore(ItemValue<Group> item, boolean isCreate) {
		// TODO Auto-generated method stub

	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void create(String uid, Group group) throws ServerFault {
		createWithExtId(uid, null, group);
	}

	@Override
	public void createWithExtId(String uid, String extId, Group group) throws ServerFault {
		DirEntryNG dirEntry = adapter.fromGroup(group);
		dirEntry.entryUid = uid;
		store.createWithId(uid, null, extId, dirEntry.displayName, dirEntry);
		System.err.println("** GROUP_NG create " + group);
	}

	@Override
	public void update(String uid, Group group) throws ServerFault {
		createWithExtId(uid, null, group);
	}

	@Override
	public void touch(String uid, boolean updateVcards) throws ServerFault {
		store.touch(uid);
	}

	@Override
	public Group get(String uid) {
		ItemValue<DirEntryNG> de = store.get(uid, null);
		return Optional.ofNullable(de).map(e -> adapter.fromEntry(e.value)).orElse(null);
	}

	@Override
	public ItemValue<Group> getComplete(String uid) throws ServerFault {
		ItemValue<DirEntryNG> de = store.get(uid, null);
		return Optional.ofNullable(de).map(e -> ItemValue.create(e, adapter.fromEntry(e.value))).orElse(null);
	}

	@Override
	public ItemValue<Group> getLight(String uid) throws ServerFault {
		return getComplete(uid);
	}

	@Override
	public ItemValue<Group> byEmail(String email) throws ServerFault {
		Item entry = store.byEmail(email);
		return Optional.ofNullable(entry).map(e -> ItemValue.create(e, get(e.uid))).orElse(null);
	}

	@Override
	public ItemValue<Group> byName(String name) throws ServerFault {
		Item entry = store.byMailboxName(GroupHelper.mailboxName(name));
		return Optional.ofNullable(entry).map(e -> ItemValue.create(e, get(e.uid))).orElse(null);
	}

	@Override
	public TaskRef delete(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public ItemValue<Group> getByExtId(String extId) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void add(String uid, List<Member> members) throws ServerFault {
		System.err.println("add(" + uid + ", " + members + ")");
		for (Member m : members) {
			store.addMemberOf(m.uid, uid);
		}
	}

	@Override
	public void remove(String uid, List<Member> members) throws ServerFault {
		System.err.println("remove(" + uid + ", " + members + ")");
		for (Member m : members) {
			store.rmMemberOf(m.uid, uid);
		}
	}

	@Override
	public List<Member> getMembers(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<Member> getExpandedMembers(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<Member> getExpandedUserMembers(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ItemValue<Group>> getParents(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<String> allUids() throws ServerFault {
		List<ItemIdAndRole> groups = store.byKind(Kind.GROUP);
		List<Long> itemIds = groups.stream().map(i -> i.itemId()).toList();
		return ServerFault.onException(
				() -> store.getItemStore().getMultipleById(itemIds).stream().map(i -> i.uid).toList(),
				ErrorCode.SQL_ERROR);
	}

	@Override
	public Set<String> getGroupsWithRoles(List<String> roles) throws ServerFault {
		Map<String, Set<Long>> idx = new HashMap<>();
		List<ItemIdAndRole> rawList = store.byKind(Kind.GROUP);
		for (var ir : rawList) {
			for (String r : ir.roles()) {
				Set<Long> cur = idx.computeIfAbsent(r, k -> new HashSet<>());
				cur.add(ir.itemId());
			}
		}
		Set<Long> found = new HashSet<>();
		for (String r : roles) {
			found.addAll(idx.getOrDefault(r, Collections.emptySet()));
		}
		try {
			return store.getItemStore().getMultipleById(List.copyOf(found)).stream().map(i -> i.uid)
					.collect(Collectors.toSet());
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public List<ItemValue<Group>> search(GroupSearchQuery query) throws ServerFault {
		// TODO Auto-generated method stub
		return Collections.emptyList();
	}

	@Override
	public Set<String> getRoles(String uid) throws ServerFault {
		Item found;
		try {
			found = store.getItemStore().get(uid);
			if (found != null) {
				return store.getRoles(found);
			} else {
				throw ServerFault.notFound("no " + uid + " group.");
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void setRoles(String uid, Set<String> roles) throws ServerFault {
		Item found;
		try {
			found = store.getItemStore().get(uid);
			if (found != null) {
				store.setRoles(found, roles);
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
