package net.bluemind.exchange.mapi.service.internal;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.annotations.VisibleForTesting;

import net.bluemind.core.backup.continuous.events.TxOutboxLookup;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.exchange.mapi.api.IMapiPidMapping;
import net.bluemind.exchange.mapi.api.MapiPidMapping;
import net.bluemind.exchange.mapi.repository.IMapiPidMappingStore;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.state.provider.IStateProvider.CloningState;
import net.bluemind.system.state.provider.StateProvider;

public class MapiPidMappingService implements IMapiPidMapping {

	public static final int START = 0x9000;
	private IMapiPidMappingStore store;

	public MapiPidMappingService(BmContext context) {
		store = RepositoryProvider.instance(IMapiPidMappingStore.class, context);
	}

	@Override
	public MapiPidMapping get(Integer id) {
		return store.get(id);
	}

	@Override
	public Integer save(String name) {
		AtomicInteger newId = new AtomicInteger();
		TxEnabler.atomically(() -> {
			newId.set(store.store(name));
			MapiPidMapping pid = new MapiPidMapping(newId.get(), name);
			pushToKafka(pid);
		});
		return newId.get();
	}

	@Override
	public List<MapiPidMapping> getAll() {
		return store.getAll();
	}

	@Override
	public void saveBatch(List<MapiPidMapping> batch) {
		TxEnabler.atomically(() -> {
			store.storeBatch(batch);
			if (StateProvider.state() != CloningState.CLONING) {
				batch.forEach(MapiPidMappingService::pushToKafka);
			}
		});
	}

	@VisibleForTesting
	public static void pushToKafka(MapiPidMapping pidMapping) {
		if (StateProvider.state() != CloningState.CLONING) {
			ItemValue<MapiPidMapping> fakeItemValue = ItemValue.create(MapiPidMapping.TYPE, pidMapping);
			fakeItemValue.internalId = pidMapping.getId();
			fakeItemValue.uid = "";
			TxOutboxLookup.forOwnerAndType(MapiPidMapping.TYPE, null, "pid_mapping_owner").forKafka(fakeItemValue, null,
					false);
		}
	}

}
