package net.bluemind.exchange.mapi.service.internal;

import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.events.TxOutboxLookup;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.exchange.mapi.api.IMapiMailboxes;
import net.bluemind.exchange.mapi.api.IMapiPCLCache;
import net.bluemind.exchange.mapi.api.KafkaPCLCacheEntry;
import net.bluemind.exchange.mapi.api.MapiPCLCacheEntry;
import net.bluemind.exchange.mapi.api.MapiReplica;
import net.bluemind.exchange.mapi.repository.IMapiPCLCacheStore;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.state.provider.IStateProvider.CloningState;
import net.bluemind.system.state.provider.StateProvider;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class MapiPCLCache implements IMapiPCLCache {

	private Logger logger = LoggerFactory.getLogger(MapiPCLCache.class);

	private IMapiPCLCacheStore store;
	BmContext context;
	private IMapiMailboxes mapiMailBoxService;
	private IContainers containerService;

	public MapiPCLCache(BmContext context) {
		store = RepositoryProvider.instance(IMapiPCLCacheStore.class, context);
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		mapiMailBoxService = provider.instance(IMapiMailboxes.class);
		containerService = provider.instance(IContainers.class);
	}

	@Override
	public MapiPCLCacheEntry get(String replicaGuid, Long globalCounter) {
		return store.get(UUID.fromString(replicaGuid), globalCounter);
	}

	@Override
	public void store(String replicaGuid, Long globalCounter, MapiPCLCacheEntry entry) {
		TxEnabler.atomically(() -> {
			store.store(UUID.fromString(replicaGuid), globalCounter, entry);
			if (StateProvider.state() != CloningState.CLONING) {
				ItemValue<KafkaPCLCacheEntry> fakeItemValue = ItemValue.create("pcl",
						new KafkaPCLCacheEntry(replicaGuid, globalCounter, entry.getModificationNanos(),
								entry.getServerReplicaGuid(), entry.getVersion(), entry.getChangekeySize()) //
				);
				fakeItemValue.internalId = globalCounter;
				getOutbox(replicaGuid).forKafka(fakeItemValue, null, false);
			}
		});
	}

	@Override
	public boolean contains(String replicaGuid, Long globalCounter) {
		return store.contains(UUID.fromString(replicaGuid), globalCounter);
	}

	@Override
	public void delete(String replicaGuid, Long globalCounter) {
		TxEnabler.atomically(() -> {
			store.delete(UUID.fromString(replicaGuid), globalCounter);
			if (StateProvider.state() != CloningState.CLONING) {
				ItemValue<KafkaPCLCacheEntry> fakeItemValue = ItemValue.create("pcl",
						new KafkaPCLCacheEntry(replicaGuid, globalCounter, null, null, null, null) //
				);
				fakeItemValue.internalId = globalCounter;
				getOutbox(replicaGuid).forKafka(fakeItemValue, null, true);
			}
		});
	}

	private ITxOutbox getOutbox(String replicaGuid) {
		MapiReplica maiboxReplica = mapiMailBoxService.byLocalReplicaGuid(replicaGuid);
		if (maiboxReplica == null) {
			throw new ServerFault("Not able to retrieve mail replica for local replica guid: " + replicaGuid);
		}
		String bluemindUser = maiboxReplica.mailboxUid;
		String containerUid = IMailboxAclUids.uidForMailbox(bluemindUser);
		ContainerDescriptor container = containerService.getIfPresent(containerUid);
		return TxOutboxLookup.forOwnerAndType(KafkaPCLCacheEntry.TYPE, container.getDomainUid(),
				maiboxReplica.mailboxUid);
	}
}
