package net.bluemind.exchange.mapi.persistence;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.sql.DataSource;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.persistence.IntegerCreator;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.exchange.mapi.api.MapiPidMapping;
import net.bluemind.exchange.mapi.repository.IMapiPidMappingStore;

public class MapiPidMappingStore extends JdbcAbstractStore implements IMapiPidMappingStore {

	private static final String SELECT = "SELECT " + MapiPidMappingColumns.cols.names()
			+ " FROM t_mapi_pidmapping WHERE pid_id=?";
	private static final String SELECT_BY_NAME = "SELECT " + MapiPidMappingColumns.cols.names()
			+ " FROM t_mapi_pidmapping WHERE pid_name=?";

	private static final String SELECT_ALL = "SELECT " + MapiPidMappingColumns.cols.names() + " FROM t_mapi_pidmapping";

	private static final String SELECT_NEXT_INSERTED_ID = "SELECT COALESCE(MAX(pid_id), 36864) FROM t_mapi_pidmapping";

	private static final String INSERT_FULL = "INSERT INTO t_mapi_pidmapping (" + MapiPidMappingColumns.cols.names()
			+ ") VALUES (" + MapiPidMappingColumns.cols.values()
			+ ") ON CONFLICT(pid_id) DO UPDATE SET pid_name = EXCLUDED.pid_name";

	private static final String INSERT_WITH_NAME = "INSERT INTO t_mapi_pidmapping (pid_name) VALUES (?) ON CONFLICT (pid_name) DO NOTHING RETURNING pid_id ";

	private static final String UPDATE_SEQ = "SELECT setval('t_mapi_pidmapping_pid_id_seq', ?, true)";

	public MapiPidMappingStore(DataSource dataSource) {
		super(dataSource);
		logger.debug("Created for ds {}", dataSource);
	}

	@Override
	public MapiPidMapping get(Integer id) throws ServerFault {
		try {
			return unique(SELECT, rs -> new MapiPidMapping(), MapiPidMappingColumns.populator(), id);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	private MapiPidMapping getByName(String name) throws ServerFault {
		try {
			return unique(SELECT_BY_NAME, rs -> new MapiPidMapping(), MapiPidMappingColumns.populator(), name);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public List<MapiPidMapping> getAll() throws ServerFault {
		try {
			return select(SELECT_ALL, rs -> new MapiPidMapping(), MapiPidMappingColumns.populator());
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public Integer store(String name) throws ServerFault {
		String query = INSERT_WITH_NAME;
		try {
			Integer result = insertAndReturn(//
					query, //
					name, //
					Arrays.asList(MapiPidMappingColumns.nameValue()), //
					IntegerCreator.FIRST, //
					null);
			if (result == null) {
				result = getByName(name).getId();
			}
			return result;
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public void storeBatch(List<MapiPidMapping> pids) throws ServerFault {
		try (Connection connection = getConnection()) {
			batchInsert(INSERT_FULL, pids, Arrays.asList(MapiPidMappingColumns.values()));
			updateNextInsertedId(connection);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	private Integer updateNextInsertedId(Connection connection) throws SQLException {
		Integer nextInsertedId = null;
		try (PreparedStatement updateSeqPreparedStatement = connection.prepareStatement(UPDATE_SEQ)) {
			nextInsertedId = unique(SELECT_NEXT_INSERTED_ID, new IntegerCreator(1), Collections.emptyList());
			updateSeqPreparedStatement.setInt(1, nextInsertedId);
			updateSeqPreparedStatement.execute();
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
		return nextInsertedId;
	}

}
