package net.bluemind.exchange.mapi.persistence.pclcache.migration;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;

import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;

public class PclCacheMigrationDoneFlag {
	private static final Logger logger = LoggerFactory.getLogger(PclCacheMigrationDoneFlag.class);
	public static final Path ROOT = Path.of("/var/spool/bm-mapi/pclcache-migration/");
	private static final String FLAG_FILE_SUFFIX = ".done";
	private static final PclCacheMigrationDoneFlag INSTANCE = new PclCacheMigrationDoneFlag();

	public static PclCacheMigrationDoneFlag get() {
		return INSTANCE;
	}

	private Set<UUID> flags = new HashSet<>();
	private boolean alldone = false;

	public PclCacheMigrationDoneFlag() {
		try {
			if (!Files.exists(ROOT)) {
				Files.createDirectories(ROOT);
			}
		} catch (IOException e) {
			logger.error("Not able to read " + ROOT.toString() + "folder", e);
		}
		rebuild();
	}

	public void markAsDone(UUID replicaGuid) {
		Path flagPath = getFlagPath(replicaGuid);
		if (!Files.exists(flagPath)) {
			try {
				Files.createFile(flagPath);
				flags.add(replicaGuid);
			} catch (IOException e) {
				logger.error("Not able to create lock file", e);
			}
		}
	}

	public void markAsAllDone() {
		alldone = true;
		clean();
		rebuild();
	}

	private void clean() {
		try {
			Files.list(ROOT).forEach(migrationFlagFile -> {
				try {
					Files.delete(migrationFlagFile);
				} catch (IOException e) {
					logger.error("Not able to clean lock file " + migrationFlagFile.toString(), e);
				}
			});
			Files.delete(ROOT);
		} catch (IOException e) {
			logger.error("Not able to clean lock files", e);
		}
	}

	public boolean isDone(UUID replicaGuid) {
		boolean isDone = flags.contains(replicaGuid);
		if (!isDone && Files.exists(getFlagPath(replicaGuid))) {
			isDone = true;
			flags.add(replicaGuid);
		}
		return isDone;
	}

	public boolean isAllDone() {
		return alldone;
	}

	@VisibleForTesting
	public void unflagAsAllDone() {
		alldone = false;
	}

	@VisibleForTesting
	public void unflagIsDone(UUID replicaGuid) {
		Path flagPath = getFlagPath(replicaGuid);
		flags.remove(replicaGuid);
		if (Files.exists(flagPath)) {
			try {
				Files.delete(flagPath);
			} catch (IOException e) {
				logger.error("Not able to create lock file", e);
			}
		}
	}

	@VisibleForTesting
	public void rebuild() {
		if (Files.exists(ROOT)) {
			try {
				flags = Files.list(ROOT)//
						.map(path -> path.getFileName().toString())//
						.filter(fileName -> fileName.endsWith(FLAG_FILE_SUFFIX))
						.map(fileName -> UUID.fromString(fileName.replaceFirst(FLAG_FILE_SUFFIX + "$", "")))//
						.collect(Collectors.toSet());
			} catch (IOException e) {
				logger.error("Not able to rebuild flags ", e);
			}
		} else {
			flags = new HashSet<>();
		}
	}

	private Path getFlagPath(UUID replicaGuid) {
		return ROOT.resolve(replicaGuid.toString() + FLAG_FILE_SUFFIX);
	}
}
