package net.bluemind.exchange.mapi.persistence;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.sql.DataSource;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.persistence.IntegerCreator;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.exchange.mapi.api.MapiPidMapping;
import net.bluemind.exchange.mapi.repository.IMapiPidMappingStore;

public class MapiPidMappingStore extends JdbcAbstractStore implements IMapiPidMappingStore {

	private static final String SELECT = "SELECT " + MapiPidMappingColumns.cols.names()
			+ " FROM t_mapi_pidmapping WHERE pid_id=?";

	private static final String SELECT_ALL = "SELECT " + MapiPidMappingColumns.cols.names() + " FROM t_mapi_pidmapping";

	private static final String SELECT_NEXT_INSERTED_ID = "SELECT COALESCE(MAX(pid_id), 36864) FROM t_mapi_pidmapping";

	private static final String INSERT_FULL = "INSERT INTO t_mapi_pidmapping (" + MapiPidMappingColumns.cols.names()
			+ ") VALUES (" + MapiPidMappingColumns.cols.values()
			+ ") ON CONFLICT(pid_id) DO UPDATE SET pid_name = EXCLUDED.pid_name";

	private static final String INSERT_WITH_NAME = "INSERT INTO t_mapi_pidmapping (pid_name) VALUES (?) RETURNING pid_id";

	private static final String UPDATE_SEQ = "SELECT setval('t_mapi_pidmapping_pid_id_seq', ?, true)";

	public MapiPidMappingStore(DataSource dataSource) {
		super(dataSource);
		logger.debug("Created for ds {}", dataSource);
	}

	@Override
	public MapiPidMapping get(Integer id) throws ServerFault {
		try {
			return unique(SELECT, rs -> new MapiPidMapping(), MapiPidMappingColumns.populator(), id);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public List<MapiPidMapping> getAll() throws ServerFault {
		try {
			return select(SELECT_ALL, rs -> new MapiPidMapping(), MapiPidMappingColumns.populator());
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public Integer store(String name) throws ServerFault {
		String query = INSERT_WITH_NAME;
		try {
			return insertAndReturn(//
					query, //
					name, //
					Arrays.asList(MapiPidMappingColumns.nameValue()), //
					IntegerCreator.FIRST, //
					null);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public Integer storeBatch(List<MapiPidMapping> pids) throws ServerFault {
		Integer nextInsertedId = null;
		try (Connection connection = getConnection()) {
			connection.setAutoCommit(false);
			try (PreparedStatement updateSeqPreparedStatement = connection.prepareStatement(UPDATE_SEQ)) {
				batchInsert(INSERT_FULL, pids, Arrays.asList(MapiPidMappingColumns.values()));

				nextInsertedId = unique(SELECT_NEXT_INSERTED_ID, new IntegerCreator(1), Collections.emptyList());
				updateSeqPreparedStatement.setInt(1, nextInsertedId);
				updateSeqPreparedStatement.execute();
				connection.commit();
			} catch (SQLException e) {
				connection.rollback();
				throw new ServerFault(e);
			}
		} catch (SQLException e) {

			throw new ServerFault(e);
		}
		return nextInsertedId;
	}
}
