package net.bluemind.exchange.mapi.persistence.pclcache.migration;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.mapdb.DB;
import org.mapdb.DBMaker;
import org.mapdb.DataInput2;
import org.mapdb.DataOutput2;
import org.mapdb.HTreeMap;
import org.mapdb.Serializer;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.exchange.mapi.api.MapiPCLCacheEntry;
import net.bluemind.exchange.mapi.repository.IMapiPCLCacheStore;

public class MapiPCLCacheMapDbStore implements IMapiPCLCacheStore {

	public static final File ROOT = new File("/var/spool/bm-mapi/pcls/");
	static {
		ROOT.mkdirs();
	}

	private static class EntrySerializer implements Serializer<MapiPCLCacheEntry> {

		public static final Serializer<MapiPCLCacheEntry> INST = new EntrySerializer();

		private static int UUID_SIZE = 16;

		@Override
		public void serialize(DataOutput2 out, MapiPCLCacheEntry value) throws IOException {
			// We just need to read from mapDb
			out.writeLong(value.getModificationNanos());
			int size = value.getChangekeySize();
			out.writeShort(size);
			serializeUuid(out, value);
			int versionSize = size - UUID_SIZE;
			out.write(serializeLongToByte(out, value.getVersion(), versionSize));
		}

		private void serializeUuid(DataOutput2 out, MapiPCLCacheEntry value) throws IOException {
			out.writeLong(value.getServerReplicaGuid().getMostSignificantBits());
			out.writeLong(value.getServerReplicaGuid().getLeastSignificantBits());
		}

		private byte[] serializeLongToByte(DataOutput2 out, long version, int size) {
			ByteBuf longBuffer = Unpooled.buffer(Long.BYTES);
			longBuffer.writeLong(version);
			longBuffer.resetReaderIndex();
			byte[] bytes = new byte[size];
			longBuffer.skipBytes(Long.BYTES - size);
			longBuffer.readBytes(bytes);
			return bytes;
		}

		@Override
		public MapiPCLCacheEntry deserialize(DataInput2 input, int available) throws IOException {
			long modifiedDate = input.readLong();
			short size = input.readShort();
			UUID uuid = deserializeUuid(input);
			long version = deserializeByteToLong(input, size - UUID_SIZE);
			return new MapiPCLCacheEntry(modifiedDate, uuid, version, size);
		}

		private UUID deserializeUuid(DataInput2 input) throws IOException {
			long mostSigBits = input.readLong();
			long leastSigBits = input.readLong();
			return new UUID(mostSigBits, leastSigBits);
		}

		private long deserializeByteToLong(DataInput2 input, int size) throws IOException {
			byte[] nBytes = new byte[size];
			input.readFully(nBytes);
			byte[] padding0Bytes = new byte[Long.BYTES - size];

			ByteBuf versionBuffer = Unpooled.buffer(Long.BYTES);
			versionBuffer.writeBytes(padding0Bytes);
			versionBuffer.writeBytes(nBytes);
			return versionBuffer.readLong();
		}
	}

	private static MapiPCLCacheMapDbStore INSTANCE = new MapiPCLCacheMapDbStore();
	private static Map<UUID, DBAndHTreeMap> MAPDB_INSTANCES = new HashMap<>();

	public static MapiPCLCacheMapDbStore get() {
		return INSTANCE;
	}

	public static record DBAndHTreeMap(DB pclsBackingStore, HTreeMap<Long, MapiPCLCacheEntry> cache) {
	}

	private MapiPCLCacheMapDbStore() {

	}

	@Override
	public void store(UUID replicaGuid, Long globalCounter, MapiPCLCacheEntry entry) throws ServerFault {
		getMapDB(replicaGuid).cache().put(globalCounter, entry);
		getMapDB(replicaGuid).pclsBackingStore().commit();
	}

	@Override
	public MapiPCLCacheEntry get(UUID replicaGuid, Long globalCounter) throws ServerFault {
		return getMapDB(replicaGuid).cache().get(globalCounter);
	}

	@Override
	public boolean contains(UUID replicaGuid, Long globalCounter) throws ServerFault {
		return getMapDB(replicaGuid).cache().containsKey(globalCounter);
	}

	@Override
	public void delete(UUID replicaGuid, Long globalCounter) {
		getMapDB(replicaGuid).cache().remove(globalCounter);
		getMapDB(replicaGuid).pclsBackingStore().commit();
	}

	public void close(UUID replicaGuid) {
		getMapDB(replicaGuid).cache().close();
		getMapDB(replicaGuid).pclsBackingStore().close();
		MAPDB_INSTANCES.remove(replicaGuid);
	}

	public DBAndHTreeMap getMapDB(UUID replicaGuid) {
		return MAPDB_INSTANCES.computeIfAbsent(replicaGuid, uuid -> {
			File context = new File(ROOT, "pcl_" + uuid.toString() + ".mapdb");
			DB pclsBackingStore = DBMaker.fileDB(context) //
					.fileMmapEnable() //
					.fileMmapPreclearDisable() //
					.cleanerHackEnable() //
					.checksumHeaderBypass() //
					.transactionEnable() //
					.make();
			HTreeMap<Long, MapiPCLCacheEntry> cache = pclsBackingStore.hashMap("pcls").keySerializer(Serializer.LONG)
					.valueSerializer(EntrySerializer.INST).createOrOpen();
			return new DBAndHTreeMap(pclsBackingStore, cache);
		});
	}
}