/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.exchange.mapi.notifications;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Verticle;
import io.vertx.core.eventbus.EventBus;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.OwnerSubscriptionsBusAddresses;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.OOPMessage;
import net.bluemind.hornetq.client.Producer;
import net.bluemind.hornetq.client.Topic;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;

public class ItemNotificationVerticle extends AbstractNotificationVerticle {

	private static final Logger logger = LoggerFactory.getLogger(ItemNotificationVerticle.class);

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {

		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new ItemNotificationVerticle();
		}

	}

	@Override
	public void start() {
		EventBus eb = vertx.eventBus();
		MQ.init(() -> {

			itemsNotificationsDispatch(eb);

			final Producer hierProducer = MQ.registerProducer(Topic.MAPI_HIERARCHY_NOTIFICATIONS);
			whenRunningBlocking(eb, Topic.MAPI_HIERARCHY_NOTIFICATIONS, hierProducer::send);

			final Producer dioProducer = MQ.registerProducer(Topic.MAPI_DELEGATION_NOTIFICATIONS);
			whenRunningBlocking(eb, Topic.MAPI_DELEGATION_NOTIFICATIONS, dioProducer::send);
			whenRunningBlocking(eb, OwnerSubscriptionsBusAddresses.ALL_SUBSCRIPTION_CHANGES, dioProducer::send);

			final Producer pfAclUpdateProducer = MQ.registerProducer(Topic.MAPI_PF_ACL_UPDATE);
			whenRunningBlocking(eb, Topic.MAPI_PF_ACL_UPDATE, pfAclUpdateProducer::send);

			final Producer daProducer = MQ.registerProducer(Topic.MAPI_DEFERRED_ACTION_NOTIFICATIONS);
			whenRunningBlocking(eb, Topic.MAPI_DEFERRED_ACTION_NOTIFICATIONS, daProducer::send);
		});

	}

	private void itemsNotificationsDispatch(EventBus eb) {
		final Producer producer = MQ.registerProducer(Topic.MAPI_ITEM_NOTIFICATIONS);
		ArrayBlockingQueue<JsonObject> notifQueue = new ArrayBlockingQueue<>(64);
		Thread.ofVirtual().name("mapi-items-notifications-vt").start(() -> {
			while (true) {
				try {
					JsonObject body = notifQueue.poll(1, TimeUnit.SECONDS);
					if (body != null) {
						OOPMessage mqMsg = prepareItemNotificationsMessage(body);
						producer.send(mqMsg);
					}
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
			}
		});

		whenRunning(eb, Topic.MAPI_ITEM_NOTIFICATIONS, body -> {
			boolean given = notifQueue.offer(body, 250, TimeUnit.MILLISECONDS);
			if (!given) {
				logger.warn("notif queue was full, skipping on item notification");
			}
		});
	}

	private OOPMessage prepareItemNotificationsMessage(JsonObject body) {
		OOPMessage mqMsg = MQ.newMessage();
		IContainers contApi = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IContainers.class);
		String contUid = body.getString("containerUid");
		BaseContainerDescriptor descriptor = contApi.getLight(contUid);
		mqMsg.putStringProperty("containerUid", contUid);
		mqMsg.putStringProperty("owner", descriptor.owner);
		mqMsg.putStringProperty("domain", descriptor.domainUid);
		mqMsg.putLongProperty("internalId", body.getLong("internalId"));
		mqMsg.putStringProperty("messageClass", body.getString("messageClass"));
		mqMsg.putStringProperty("operation", body.getString("operation"));
		return mqMsg;
	}

}
