/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.verticle;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Handler;
import io.vertx.core.eventbus.Message;
import io.vertx.core.json.JsonObject;
import net.bluemind.addressbook.api.AddressBookBusAddresses;
import net.bluemind.calendar.hook.CalendarHookAddress;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.OOPMessage;
import net.bluemind.hornetq.client.Producer;
import net.bluemind.hornetq.client.Topic;
import net.bluemind.lib.vertx.utils.ThrottleAccumulator;
import net.bluemind.system.state.provider.IStateProvider.CloningState;
import net.bluemind.system.state.provider.StateProvider;
import net.bluemind.todolist.hook.TodoListHookAddress;

public class EasContainerChangeVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(EasContainerChangeVerticle.class);
	private Producer calendarProducer;
	private Producer addressbookProducer;
	private Producer todolistProducer;

	@Override
	public void start() {
		MQ.init(() -> {
			calendarProducer = MQ.registerProducer(Topic.CALENDAR_NOTIFICATIONS);
			addressbookProducer = MQ.registerProducer(Topic.CONTACT_NOTIFICATIONS);
			todolistProducer = MQ.registerProducer(Topic.TASK_NOTIFICATIONS);
		});

		Handler<Message<JsonObject>> calendarChangedHandler = (Message<JsonObject> event) -> {
			if (calendarProducer != null) {
				OOPMessage msg = buildMessage(event);
				calendarProducer.send(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("Wake up {} devices for calendar changes", event.body().getString("loginAtDomain"));
				}
			} else {
				logger.warn("no calendar change notification, failed to create producer");
			}
		};
		Handler<Message<JsonObject>> addressbookChangedHandler = (Message<JsonObject> event) -> {
			if (addressbookProducer != null) {
				OOPMessage msg = buildMessage(event);
				addressbookProducer.send(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("Wake up {} devices for contacts changes", event.body().getString("loginAtDomain"));
				}
			} else {
				logger.warn("no contacts change notification, failed to create producer");
			}
		};
		Handler<Message<JsonObject>> todolistChangedHandler = (Message<JsonObject> event) -> {
			if (todolistProducer != null) {
				OOPMessage msg = buildMessage(event);
				todolistProducer.send(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("Wake up {} devices for todolist changes", event.body().getString("loginAtDomain"));
				}
			} else {
				logger.warn("no todolist change notification, failed to create producer");
			}
		};

		ThrottleAccumulator<JsonObject> throttleCalendar = new ThrottleAccumulator<>(vertx, Duration.ofSeconds(10),
				msg -> msg.body().getString("container"), calendarChangedHandler);
		ThrottleAccumulator<JsonObject> throttleAddressBook = new ThrottleAccumulator<>(vertx, Duration.ofSeconds(10),
				msg -> msg.body().getString("container"), addressbookChangedHandler);
		ThrottleAccumulator<JsonObject> throttleTodoList = new ThrottleAccumulator<>(vertx, Duration.ofSeconds(10),
				msg -> msg.body().getString("container"), todolistChangedHandler);

		vertx.eventBus().consumer(CalendarHookAddress.CHANGED,
				(Message<JsonObject> e) -> onlyIfStateRunning(e, throttleCalendar::handle));
		vertx.eventBus().consumer(AddressBookBusAddresses.CHANGED,
				(Message<JsonObject> e) -> onlyIfStateRunning(e, throttleAddressBook::handle));
		vertx.eventBus().consumer(TodoListHookAddress.CHANGED,
				(Message<JsonObject> e) -> onlyIfStateRunning(e, throttleTodoList::handle));
	}

	private void onlyIfStateRunning(Message<JsonObject> event, Handler<Message<JsonObject>> handler) {
		if (StateProvider.state() == CloningState.NOT_CLONING) {
			handler.handle(event);
		}
	}

	private OOPMessage buildMessage(Message<JsonObject> event) {
		OOPMessage msg = MQ.newMessage();
		msg.putStringProperty("container", event.body().getString("container"));
		msg.putStringProperty("userUid", event.body().getString("loginAtDomain"));
		msg.putStringProperty("domainUid", event.body().getString("domainUid"));
		return msg;
	}

}
