package net.bluemind.eas.validation;

import java.io.File;
import java.io.FileOutputStream;

import javax.xml.transform.dom.DOMSource;
import javax.xml.validation.Validator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import net.bluemind.eas.utils.DOMUtils;
import net.bluemind.eas.validation.impl.ValidatorFactory;
import net.bluemind.eas.validation.impl.ValidatorFactory.Direction;
import net.bluemind.eas.validation.impl.ValidatorFactory.ProtoVersion;

public class SchemaBasedValidator implements IProtocolValidator {

	private static final Logger logger = LoggerFactory.getLogger(SchemaBasedValidator.class);

	@Override
	public void checkRequest(double protocolVersion, Document doc) throws ValidationException {
		Element root = doc.getDocumentElement();
		logger.debug("Root elem: {}, name: {}", root, root.getNodeName());
		String cmd = root.getNodeName();
		logger.debug("Checking {} AS request with XSDs v{}...", cmd, protocolVersion);
		ProtoVersion pv = protocolVersion < 14 ? ProtoVersion.v12
				: protocolVersion < 16 ? ProtoVersion.v14 : ProtoVersion.v16;

		if (protocolVersion > 14) {
			// don't know where to get the xsd for 16.x
			return;
		}

		Validator validator = ValidatorFactory.createFor(pv, Direction.request, cmd);
		if (validator != null) {
			try {
				validator.validate(new DOMSource(doc.getDocumentElement()));
			} catch (Exception e) {
				throw new ValidationException(e.getMessage());
			}
		}
	}

	@Override
	public void checkResponse(double protocolVersion, Document doc) throws ValidationException {
		String cmd = doc.getDocumentElement().getLocalName();
		logger.info("Checking {} AS response with XSDs v{}...", cmd, protocolVersion);
		ProtoVersion pv = protocolVersion < 14 ? ProtoVersion.v12
				: protocolVersion < 16 ? ProtoVersion.v14 : ProtoVersion.v16;

		if (protocolVersion > 14) {
			// don't know where to get the xsd for 16.x
			return;
		}

		Validator validator = ValidatorFactory.createFor(pv, Direction.response, cmd);
		if (validator != null) {
			try {
				validator.validate(new DOMSource(doc.getDocumentElement()));
			} catch (Exception e) {
				String msg = e.getMessage();
				try {
					File invalid = File.createTempFile("eas_invalid", ".xml");
					DOMUtils.serialise(doc, new FileOutputStream(invalid), true);
					msg = "Invalid XML saved to " + invalid.getAbsolutePath() + " " + msg;
				} catch (Exception e1) {
					logger.error(e.getMessage(), e);
				}
				throw new ValidationException(msg);
			}
		}
	}

}
