/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.service.internal;

import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import com.google.common.hash.Hashing;

import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.eas.api.Account;
import net.bluemind.eas.api.FolderSyncVersions;
import net.bluemind.eas.api.Heartbeat;
import net.bluemind.eas.api.IEas;
import net.bluemind.eas.api.IInCoreEas;
import net.bluemind.eas.persistence.EasStore;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.Topic;
import net.bluemind.role.api.BasicRoles;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class EasService implements IEas, IInCoreEas {

	private EasStore store;
	private BmContext context;
	private ITxOutbox heatbeatOutbox;
	private ITxOutbox resetOutbox;
	private ITxOutbox folderSyncOutbox;
	private ITxOutbox clientIdOutbox;

	public EasService(BmContext context) {
		this.context = context;
		store = new EasStore(context.getDataSource());
		IServiceProvider provider = context.su().provider();
		this.clientIdOutbox = provider.instance(ITxOutbox.class, "global.virt", "system", "easclientid", "easclientid",
				DataLocation.directory().serverUid());
		this.heatbeatOutbox = provider.instance(ITxOutbox.class, "global.virt", "system", "easheatbeat", "easheatbeat",
				DataLocation.directory().serverUid());
		this.resetOutbox = provider.instance(ITxOutbox.class, "global.virt", "system", "easreset", "easreset",
				DataLocation.directory().serverUid());
		this.folderSyncOutbox = provider.instance(ITxOutbox.class, "global.virt", "system", "easfoldersync",
				"easfoldersync", DataLocation.directory().serverUid());
	}

	@Override
	public Heartbeat getHeartbeat(String deviceUid) throws ServerFault {
		checkAccess();
		try {
			return store.getHeartbeat(deviceUid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	private Item item(String sid) {
		String kUid = context.getSecurityContext().getSubject() + "-" + sid;
		return Item.create(kUid, Hashing.sipHash24().hashString(kUid, StandardCharsets.US_ASCII).asLong());
	}

	@Override
	public void setHeartbeat(Heartbeat heartbeat) throws ServerFault {
		checkAccess();
		TxEnabler.atomically(() -> {
			store.setHeartbeat(heartbeat);
			heatbeatOutbox.forKafka(ItemValue.create(item(heartbeat.deviceUid), heartbeat), null, false);
		});
	}

	@Override
	public Boolean needReset(Account account) throws ServerFault {
		checkAccess();
		try {
			return store.needReset(account);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void deletePendingReset(Account account) throws ServerFault {
		checkAccess();
		TxEnabler.atomically(() -> {
			try {
				store.deletePendingReset(account);
				resetOutbox.forKafka(ItemValue.create(item(account.device), account), null, true);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});
	}

	@Override
	public void insertPendingReset(Account account) throws ServerFault {
		RBACManager rbacManager = new RBACManager(context).forContainer("device:" + account.userUid)
				.forEntry(account.userUid);
		rbacManager.check(BasicRoles.ROLE_MANAGE_USER_DEVICE);
		store.insertPendingReset(account);
		TxEnabler.atomically(() -> {
			store.insertPendingReset(account);
			resetOutbox.forKafka(ItemValue.create(item(account.device), account), null, false);
			TxEnabler.durableStorageAction(() -> MQ.getProducer(Topic.EAS_RESET_SYNC)
					.send(new JsonObject().put("user", account.userUid).put("device", account.device)));
		});
	}

	@Override
	public void insertClientId(String clientId) throws ServerFault {
		checkAccess();
		TxEnabler.atomically(() -> {
			try {
				store.insertClientId(clientId);
				clientIdOutbox.forKafka(ItemValue.create(item(clientId), clientId), null, false);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});
	}

	@Override
	public Boolean isKnownClientId(String clientId) throws ServerFault {
		checkAccess();
		boolean ret = true;
		try {
			ret = store.isKnownClientId(clientId);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		return ret;
	}

	@Override
	public void setFolderSyncVersions(FolderSyncVersions versions) throws ServerFault {
		checkAccess();
		TxEnabler.atomically(() -> {
			store.setFolderSyncVersions(versions.account, versions.versions);
			folderSyncOutbox.forKafka(ItemValue.create(item(versions.account.device), versions), null, false);
		});
	}

	@Override
	public Map<String, String> getFolderSyncVersions(Account account) throws ServerFault {
		checkAccess();
		try {
			return store.getFolderSyncVersions(account);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public Map<String, String> getConfiguration() throws ServerFault {
		if (context.getSecurityContext().isAnonymous()) {
			throw new ServerFault("Permission denied", ErrorCode.PERMISSION_DENIED);
		}
		ISystemConfiguration service = context.su().provider().instance(ISystemConfiguration.class);

		SystemConf systemConf = service.getValues();

		HashMap<String, String> ret = new HashMap<>();
		ret.put(SysConfKeys.eas_sync_unknown.name(), systemConf.stringValue(SysConfKeys.eas_sync_unknown.name()));
		ret.put(SysConfKeys.eas_min_heartbeat.name(), systemConf.stringValue(SysConfKeys.eas_min_heartbeat.name()));
		ret.put(SysConfKeys.eas_max_heartbeat.name(), systemConf.stringValue(SysConfKeys.eas_max_heartbeat.name()));

		return ret;
	}

	private void checkAccess() throws ServerFault {
		if (!context.getSecurityContext().isDomainAdmin(context.getSecurityContext().getContainerUid())) {
			throw new ServerFault("Permission denied", ErrorCode.PERMISSION_DENIED);
		}
	}

	/**
	 * Internal API, only used to populate all device stores into the kafka store
	 * initially NO security checks here, because it's internal
	 */
	@Override
	public void resyncAllToKafka() {
		try {
			store.getAllClientIds().forEach(
					clientId -> clientIdOutbox.forKafka(ItemValue.create(item(clientId), clientId), null, false));
			store.getAllHeatbeat().forEach(heartbeat -> heatbeatOutbox
					.forKafka(ItemValue.create(item(heartbeat.deviceUid), heartbeat), null, false));
			store.getAllPendingReset().forEach(rs -> resetOutbox
					.forKafka(ItemValue.create(item(rs.device), Account.create(rs.account, rs.device)), null, false));
			store.getAllFolderSyncVersions().forEach(
					fs -> folderSyncOutbox.forKafka(ItemValue.create(item(fs.account.device), fs), null, false));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}
}
