/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.domain.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import net.bluemind.core.api.Regex;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.DomainSettingsKeys;
import net.bluemind.domain.api.IDomains;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.system.api.SysConfKeys;

public class DomainUrlValidator {
	private static final String GLOBAL_VIRT = "global.virt";

	private final String domainUid;
	private final String newUrl;
	private final List<String> newOtherUrls;

	public static boolean isSet(String newUrl) {
		return newUrl != null && !newUrl.trim().isEmpty();
	}

	public static boolean isUpdated(String newUrl, String oldUrl, String newOtherUrls, String oldOtherUrls) {
		return !Objects.equals(newUrl == null ? "" : newUrl.trim(), oldUrl == null ? "" : oldUrl.trim())
				|| !Objects.equals(newOtherUrls == null ? "" : newOtherUrls.trim(),
						oldOtherUrls == null ? "" : oldOtherUrls.trim());
	}

	public static void forGlobalUrl(String newUrl, String oldUrl, String newOtherUrls, String oldOtherUrls) {
		if (!isUrlSet(newUrl)) {
			throw new ServerFault("Global external URL must not be null or empty!", ErrorCode.INVALID_PARAMETER);
		}

		if (!isUpdated(newUrl, oldUrl, newOtherUrls, oldOtherUrls)) {
			return;
		}

		new DomainUrlValidator(GLOBAL_VIRT, newUrl, newOtherUrls).check();
	}

	public static void forDomainUrl(String domainUid, String newUrl, String newOtherUrls) {
		if (domainUid == null || domainUid.isBlank()) {
			throw new ServerFault("Domain UID must be set", ErrorCode.INVALID_PARAMETER);
		}

		if (!isUrlSet(newUrl) && isUrlSet(newOtherUrls)) {
			throw new ServerFault("Domain other URLs must not be set if external URL is not set",
					ErrorCode.INVALID_PARAMETER);
		}

		new DomainUrlValidator(domainUid, newUrl, newOtherUrls).check();
	}

	private static boolean isUrlSet(String url) {
		return !(url == null || url.isBlank());
	}

	private DomainUrlValidator(String domainUid, String newUrl, String newOtherUrls) {
		this.domainUid = domainUid;
		this.newUrl = sanitizeUrl(newUrl);
		this.newOtherUrls = sanitizeOtherUrls(newOtherUrls);
	}

	private String sanitizeUrl(String url) {
		return url == null || url.isBlank() ? null : url;
	}

	private List<String> sanitizeOtherUrls(String otherUrls) {
		if (otherUrls == null || otherUrls.isBlank()) {
			return Collections.emptyList();
		}

		return Arrays.asList(otherUrls.split(" ")).stream().map(this::sanitizeUrl).filter(Objects::nonNull).toList();
	}

	private List<String> allUrls() {
		List<String> allUrls = new ArrayList<>(newOtherUrls);
		if (newUrl != null) {
			allUrls.add(newUrl);
		}

		return allUrls;
	}

	private void check() {
		if (!isUrlSet(newUrl) && newOtherUrls.isEmpty()) {
			// No URL set
			return;
		}

		if (!Regex.DOMAIN.validate(newUrl)) {
			throw new ServerFault("Invalid domain external URL: " + newUrl, ErrorCode.INVALID_PARAMETER);
		}

		List<String> invalidOtherUrls = newOtherUrls.stream().filter(url -> !Regex.DOMAIN.validate(url)).toList();
		if (!invalidOtherUrls.isEmpty()) {
			throw new ServerFault("Invalid domain other external URLs:" + String.join(", ", invalidOtherUrls),
					ErrorCode.INVALID_PARAMETER);
		}

		List<String> domainUrls = allUrls();
		Set<String> duplicatedUrl = duplicatedUrl(domainUrls, new HashSet<>());
		if (!duplicatedUrl.isEmpty()) {
			throw new ServerFault("Duplicated domain URLs: " + String.join(", ", duplicatedUrl),
					ErrorCode.INVALID_PARAMETER);
		}

		getOtherDomainsUrlsByDomainUid().entrySet()
				.forEach(otherDomainUrls -> checkDuplicateWithOtherDomainUrls(domainUrls, otherDomainUrls));
	}

	private Map<String, Set<String>> getOtherDomainsUrlsByDomainUid() {
		Map<String, Set<String>> urlsByDomainUid = new HashMap<>();

		if (!domainUid.equals(GLOBAL_VIRT)) {
			urlsByDomainUid.put(GLOBAL_VIRT,
					allSanitzedUrls(GLOBAL_VIRT, Shared.mapSysconf().get(SysConfKeys.external_url.name()),
							Shared.mapSysconf().get(SysConfKeys.other_urls.name())));
		}

		SharedMap<String, Map<String, String>> domainSettingsSharedMap = Shared.mapDomainSettings();
		domainSettingsSharedMap.keys().stream().filter(du -> !du.equals(domainUid))
				.filter(du -> !du.equals(GLOBAL_VIRT)) // global.virt URLs are in SystemConfiguration
				.forEach(du -> urlsByDomainUid.put(du,
						allSanitzedUrls(du, domainSettingsSharedMap.get(du).get(DomainSettingsKeys.external_url.name()),
								domainSettingsSharedMap.get(du).get(DomainSettingsKeys.other_urls.name()))));

		return urlsByDomainUid;
	}

	private Set<String> allSanitzedUrls(String domainUid, String url, String otherUrls) {
		return new HashSet<>(new DomainUrlValidator(domainUid, url, otherUrls).allUrls());
	}

	private void checkDuplicateWithOtherDomainUrls(List<String> domainUrls,
			Entry<String, Set<String>> otherDomainUrls) {
		Set<String> domainDuplicatedUrls = duplicatedUrl(domainUrls, otherDomainUrls.getValue());

		if (domainDuplicatedUrls.isEmpty()) {
			return;
		}

		if (otherDomainUrls.getKey().equals(GLOBAL_VIRT)) {
			throw new ServerFault(
					"External URL already used as global external URL: " + String.join(", ", domainDuplicatedUrls),
					ErrorCode.INVALID_PARAMETER);
		}

		String conflictingDomainDefaultAlias = otherDomainUrls.getKey();
		try {
			conflictingDomainDefaultAlias = Optional
					.ofNullable(ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IDomains.class)
							.get(otherDomainUrls.getKey()))
					.map(d -> d.value.defaultAlias).orElse(otherDomainUrls.getKey());
		} catch (ServerFault | NullPointerException e) {
			// Ignore and continue, use domain UID instead of domain default alias in
			// ServerFault message thrown below
			// NullPointerException used for JUnit
		}

		throw new ServerFault("External URL already used as URL for domain: " + conflictingDomainDefaultAlias + " - "
				+ String.join(", ", domainDuplicatedUrls), ErrorCode.INVALID_PARAMETER);
	}

	private Set<String> duplicatedUrl(Collection<String> urls, Set<String> uniqueUrls) {
		Set<String> duplicatedUrl = new HashSet<>();

		urls.stream().filter(url -> !uniqueUrls.add(url)).forEach(duplicatedUrl::add);

		return duplicatedUrl;
	}
}
