/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.domain.service.internal;

import java.time.DateTimeException;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.DomainSettingsKeys;
import net.bluemind.domain.api.IDomains;
import net.bluemind.domain.service.DomainUrlValidator;
import net.bluemind.keycloak.utils.AuthConfigHelper;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox.Routing;

public class DomainSettingsValidator {
	private final Logger logger = LoggerFactory.getLogger(DomainSettingsValidator.class);

	public void create(BmContext context, Map<String, String> settings, String domainUid) throws ServerFault {
		checkSplitDomain(settings);
		checkDomainMaxUsers(settings);

		if (DomainUrlValidator.isSet(settings.get(DomainSettingsKeys.external_url.name()))
				|| DomainUrlValidator.isSet(settings.get(DomainSettingsKeys.other_urls.name()))) {
			if (!context.getSecurityContext().isDomainGlobal()) {
				throw new ServerFault("Only global admin can update domain external URL", ErrorCode.FORBIDDEN);
			}

			DomainUrlValidator.forDomainUrl(domainUid, settings.get(DomainSettingsKeys.external_url.name()),
					settings.get(DomainSettingsKeys.other_urls.name()));
		}

		checkDefaultDomain(context, domainUid,
				Optional.ofNullable(settings.get(DomainSettingsKeys.default_domain.name()))
						.map(dd -> dd.isEmpty() ? null : dd));
		checkDates(settings);
		checkLanguage(settings);
		AuthConfigHelper.checkSettings(context, domainUid, settings);
	}

	public void update(BmContext context, Map<String, String> oldSettings, Map<String, String> newSettings,
			String domainUid) throws ServerFault {
		checkSplitDomain(newSettings);

		if (null != getRelay(oldSettings) && (null == getRelay(newSettings))) {
			List<String> externalUids = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IMailboxes.class, domainUid).byRouting(Routing.external);
			if (!externalUids.isEmpty()) {
				logger.error("{} Routing.external user(s)", externalUids.size());
				throw new ServerFault("Cannot unset split relay. Some mailboxes are still Routing.external",
						ErrorCode.INVALID_PARAMETER);
			}

		}

		checkDomainMaxUsers(newSettings);

		if (DomainUrlValidator.isUpdated(newSettings.get(DomainSettingsKeys.external_url.name()),
				oldSettings.get(DomainSettingsKeys.external_url.name()),
				newSettings.get(DomainSettingsKeys.other_urls.name()),
				oldSettings.get(DomainSettingsKeys.other_urls.name()))) {
			if (!context.getSecurityContext().isDomainGlobal()) {
				throw new ServerFault("Only global admin can update domain external URL", ErrorCode.FORBIDDEN);
			}

			DomainUrlValidator.forDomainUrl(domainUid, newSettings.get(DomainSettingsKeys.external_url.name()),
					newSettings.get(DomainSettingsKeys.other_urls.name()));
		}

		checkDefaultDomain(context, domainUid,
				Optional.ofNullable(newSettings.get(DomainSettingsKeys.default_domain.name()))
						.map(dd -> dd.isEmpty() ? null : dd));
		checkDates(newSettings);
		checkLanguage(newSettings);
		AuthConfigHelper.checkSettings(context, domainUid, newSettings);
	}

	private void checkSplitDomain(Map<String, String> settings) throws ServerFault {
		if (isForwardUnknownToRelay(settings) && null == getRelay(settings)) {
			throw new ServerFault("Split domain relay hostname cannot be empty", ErrorCode.INVALID_PARAMETER);
		}
	}

	private void checkLanguage(Map<String, String> settings) {
		if (settings.containsKey(DomainSettingsKeys.lang.name())) {
			String lang = settings.get(DomainSettingsKeys.lang.name()).toLowerCase();
			if (!(lang.equals("de") || lang.equals("en") || lang.equals("es") || lang.equals("fr") || lang.equals("it")
					|| lang.equals("pl") || lang.equals("sl") || lang.equals("zh") || lang.equals("hu"))) {
				throw new ServerFault("Languages " + lang + " is not supported", ErrorCode.INVALID_PARAMETER);
			}
		}

	}

	private void checkDates(Map<String, String> settings) {
		if (settings.containsKey(DomainSettingsKeys.date.name())) {
			String datePattern = settings.get(DomainSettingsKeys.date.name());
			if (datePattern == null || datePattern.isBlank()) {
				throw new ServerFault("Blank date pattern is invalid", ErrorCode.INVALID_PARAMETER);
			}
			try {
				DateTimeFormatter.ofPattern(datePattern);
				for (char c : datePattern.toCharArray()) {
					if (Character.isLetter(c) && !(c == 'y' || c == 'M' || c == 'd')) {
						throw new IllegalArgumentException("pattern contains invalid temporal chars");
					}
				}
			} catch (IllegalArgumentException e) {
				throw new ServerFault("Date pattern " + datePattern + " is invalid: " + e.getMessage(),
						ErrorCode.INVALID_PARAMETER);
			}
		}

		if (settings.containsKey(DomainSettingsKeys.timeformat.name())) {
			String timePattern = settings.get(DomainSettingsKeys.timeformat.name());
			if (timePattern == null || timePattern.isBlank()) {
				throw new ServerFault("Blank time pattern is invalid", ErrorCode.INVALID_PARAMETER);
			}
			try {
				DateTimeFormatter.ofPattern(timePattern);
				for (char c : timePattern.toCharArray()) {
					if (Character.isLetter(c)
							&& !(c == 'h' || c == 'H' || c == 'k' || c == 'K' || c == 'm' || c == 's' || c == 'a')) {
						throw new IllegalArgumentException("pattern contains invalid temporal chars");
					}
				}
			} catch (IllegalArgumentException e) {
				throw new ServerFault("Time pattern " + timePattern + " is invalid: " + e.getMessage(),
						ErrorCode.INVALID_PARAMETER);
			}
		}

		if (settings.containsKey(DomainSettingsKeys.timezone.name())) {
			String timeZone = settings.get(DomainSettingsKeys.timezone.name());
			try {
				ZoneId.of(timeZone);
			} catch (DateTimeException e) {
				throw new ServerFault("Zone ID " + timeZone + " is invalid: " + e.getMessage(),
						ErrorCode.INVALID_PARAMETER);
			}
		}

	}

	private boolean isForwardUnknownToRelay(Map<String, String> settings) {
		return Boolean.parseBoolean(settings.get(DomainSettingsKeys.mail_forward_unknown_to_relay.name()));
	}

	private String getRelay(Map<String, String> settings) {
		boolean notNull = settings.get(DomainSettingsKeys.mail_routing_relay.name()) != null;
		boolean empty = true;
		if (notNull) {
			empty = settings.get(DomainSettingsKeys.mail_routing_relay.name()).trim().isEmpty();
		}
		if (!empty) {
			return settings.get(DomainSettingsKeys.mail_routing_relay.name());
		}
		return null;
	}

	private void checkDomainMaxUsers(Map<String, String> settings) throws ServerFault {
		String domainMaxUser = settings.get(DomainSettingsKeys.domain_max_users.name());
		if (domainMaxUser == null || domainMaxUser.isEmpty()) {
			return;
		}

		Integer maxUsers = null;
		try {
			maxUsers = Integer.parseInt(domainMaxUser);
		} catch (NumberFormatException nfe) {
			throw new ServerFault("Invalid maximum number of users. Must be an integer greater than 0.",
					ErrorCode.INVALID_PARAMETER);
		}

		if (maxUsers < 1) {
			throw new ServerFault("Invalid maximum number of users. Must be an integer greater than 0.",
					ErrorCode.INVALID_PARAMETER);
		}
	}

	private void checkDefaultDomain(BmContext context, String domainUid, Optional<String> defaultDomain) {
		if (!defaultDomain.isPresent()) {
			return;
		}

		ItemValue<Domain> domain = defaultDomain.map(dd -> getDomainItemValue(context, dd))
				.orElseThrow(() -> new ServerFault(
						String.format("default domain '%s' not found", defaultDomain.orElse(null)),
						ErrorCode.INVALID_PARAMETER));

		if (!domainUid.equals(domain.uid)) {
			throw new ServerFault(
					String.format("Default domain '%s' is not an alias of domain uid '%s'", defaultDomain, domainUid),
					ErrorCode.INVALID_PARAMETER);
		}
	}

	public ItemValue<Domain> getDomainItemValue(BmContext context, String domain) {
		try {
			return ServerSideServiceProvider.getProvider(context).instance(IDomains.class).findByNameOrAliases(domain);
		} catch (Exception e) {
			logger.error("unable to retrieve domain uid: {}", domain, e);
		}

		return null;
	}
}
