package net.bluemind.directory.service.internal;

import java.util.List;
import java.util.Set;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IInternalContainerManagement;
import net.bluemind.core.container.hooks.ContainersHookAdapter;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.IMailboxAclUids;

/**
 * Hook that applies SHARED_MAILBOX ACLs to newly created containers.
 */
public class SharedMailboxAclHook extends ContainersHookAdapter {

	private static Set<String> skipContainerTypes = Set.of(IMailboxAclUids.TYPE, "replicated_mailboxes",
			"mailbox_records", "owner_subscriptions");

	@Override
	public void onContainerCreated(BmContext ctx, ContainerDescriptor cd) throws ServerFault {
		if (skipContainerTypes.contains(cd.type)) {
			return;
		}
		DirEntry ownerEntry = ctx.provider().instance(IDirectory.class, cd.domainUid).findByEntryUid(cd.owner);
		if (ownerEntry.kind != Kind.SHARED_MAILBOX) {
			return;
		}

		String uidForMailbox = IMailboxAclUids.uidForMailbox(ownerEntry.entryUid);
		IContainerManagement containerMgmtService = ctx.provider().instance(IContainerManagement.class, uidForMailbox);
		List<AccessControlEntry> mailboxAcls = containerMgmtService.getAccessControlList().stream()
				.filter(ac -> !ac.subject.equals(cd.owner)).toList();

		containerMgmtService = ctx.provider().instance(IContainerManagement.class, cd.uid);
		List<AccessControlEntry> containerAcls = containerMgmtService.getAccessControlList();

		containerAcls.addAll(mailboxAcls);
		ctx.provider().instance(IInternalContainerManagement.class, cd.uid).setAccessControlList(containerAcls, false);
	}

}
