/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.service.internal;

import java.sql.SQLException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.persistence.VCardStore;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IChangelogStore.LogEntry;
import net.bluemind.core.container.repository.ICustomDecorator;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.directory.api.BaseDirEntry.AccountType;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.persistence.DirEntryStore;
import net.bluemind.directory.persistence.ManageableOrgUnit;
import net.bluemind.directory.persistence.OrgUnitStore;
import net.bluemind.directory.service.BaseDirStoreService;
import net.bluemind.document.storage.DocumentStorage;
import net.bluemind.document.storage.IDocumentStore;

public class DirEntryStoreService extends BaseDirStoreService<DirEntry> {

	public enum CustomDecorator implements ICustomDecorator {
		ORG_UNIT
	}

	private static final Logger logger = LoggerFactory.getLogger(DirEntryStoreService.class);
	private DirEntryStore entryStore;
	private VCardStore vcardStore;
	private IDocumentStore documentStore;
	private OrgUnitStore ouStore;
	private final String domainUid;
	private DirEntriesCache cache;

	public DirEntryStoreService(BmContext context, Container container, String domainUid) {
		super(context, context.getDataSource(), context, container,
				new DirEntryStore(context.getDataSource(), container));
		this.ouStore = new OrgUnitStore(context.getDataSource(), container);
		this.entryStore = new DirEntryStore(context.getDataSource(), container);
		this.vcardStore = new VCardStore(context.getDataSource(), container);
		this.domainUid = domainUid;
		documentStore = DocumentStorage.store;
		this.cache = DirEntriesCache.get(context, domainUid);
	}

	public ListResult<ItemValue<DirEntry>> searchManageable(DirEntryQuery query, List<ManageableOrgUnit> manageable) {
		ListResult<Item> items = searchManageableItems(query, manageable);

		ListResult<ItemValue<DirEntry>> ret = new ListResult<>();
		ret.total = items.total;
		ret.values = getItemsValue(items.values);
		return ret;
	}

	public ListResult<Item> searchManageableItems(DirEntryQuery query, List<ManageableOrgUnit> manageable) {
		try {
			return entryStore.searchManageable(query, manageable);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public ListResult<ItemValue<DirEntry>> search(DirEntryQuery query, boolean light) throws ServerFault {
		ListResult<ItemValue<DirEntry>> ret = new ListResult<>();
		try {
			ListResult<Item> items = entryStore.search(query);
			ret.total = items.total;
			if (!light) {
				ret.values = getItemsValue(items.values);
			} else {
				ret.values = getItemsUids(items.values);
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

		return ret;
	}

	private List<ItemValue<DirEntry>> getItemsUids(List<Item> values) {
		return values.stream().map(v -> ItemValue.create(v, new DirEntry())).toList();
	}

	public List<ItemValue<DirEntry>> getEntries(String path) throws ServerFault {
		List<String> uids = null;
		try {
			uids = entryStore.path(path);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

		return getMultiple(uids);
	}

	public ItemValue<VCard> getVCard(String uid) throws ServerFault {
		return doOrFail(() -> {
			Item item = itemStore.get(uid);
			if (item == null) {
				return null;
			}
			return ItemValue.create(item, vcardStore.get(item));
		});
	}

	public void setPhoto(String uid, byte[] photo, byte[] icon) throws ServerFault {
		atomic(() -> {
			Item item = itemStore.get(uid);
			if (item == null) {
				throw ServerFault.notFound("entry[" + uid + "]@" + container.uid + " not found");
			}
			item = itemStore.update(uid, item.displayName);
			entryStore.setHasPhotoIcon(uid, photo, icon);
			if (hasChangeLog) {
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), securityContext.getOrigin(), item.id, 0));
			}
			TxEnabler.durableStorageAction(() -> {
				if (photo != null && photo.length > 0) {
					documentStore.store(getPhotoUid(uid), photo);
				}
				if (icon != null && icon.length > 0) {
					documentStore.store(getIconUid(uid), icon);
				}
			});
			return null;
		});
	}

	public byte[] getPhoto(String uid) throws ServerFault {
		if (hasPhoto(uid)) {
			return documentStore.get(getPhotoUid(uid));
		} else {
			return new byte[0];
		}
	}

	public boolean hasPhoto(String uid) throws ServerFault {
		return doOrFail(() -> entryStore.hasPhoto(uid));
	}

	public byte[] getIcon(String uid) throws ServerFault {
		if (hasIcon(uid)) {
			return documentStore.get(getIconUid(uid));
		} else {
			return new byte[0];
		}
	}

	public boolean hasIcon(String uid) throws ServerFault {
		return doOrFail(() -> entryStore.hasIcon(uid));
	}

	private String getPhotoUid(String uid) {
		return "dir_" + container.uid + "/photos/" + uid;
	}

	private String getIconUid(String uid) {
		return "dir_" + container.uid + "/icons/" + uid;
	}

	@Override
	protected void decorate(Item item, ItemValue<DirEntry> value, List<? extends ICustomDecorator> customDecorator)
			throws ServerFault {
		if (value.value == null) {
			logger.warn("[{}] no direntry for '{}' !!", domainUid, item.uid);
			return;
		}
		if (value.value.orgUnitUid != null && CustomDecorator.ORG_UNIT.shouldApply(customDecorator)) {
			try {
				value.value.orgUnitPath = ouStore.getPathByUid(value.value.orgUnitUid);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		}
	}

	@Override
	protected void decorate(Item item, ItemValue<DirEntry> value) throws ServerFault {
		decorate(item, value, List.of(CustomDecorator.ORG_UNIT));
	}

	@Override
	public ItemVersion update(String uid, String displayName, DirEntry value) throws ServerFault {
		ItemVersion updated = super.update(uid, displayName, value);
		cache.invalidate(uid);
		return updated;
	}

	@Override
	public ItemVersion delete(String uid) throws ServerFault {
		ItemVersion deleted = super.delete(uid);
		cache.invalidate(uid);
		return deleted;
	}

	@Override
	public void deleteAll() throws ServerFault {
		cache.invalidateAll();
		super.deleteAll();
	}

	public ItemValue<DirEntry> getByEmail(String email, boolean isDomainEmail) {
		return doOrFail(() -> {
			String res = entryStore.byEmail(email, isDomainEmail);
			if (res != null) {
				return get(res, null);
			} else {
				return null;
			}
		});
	}

	public void updateAccountType(String uid, AccountType accountType) throws ServerFault {
		try {
			Item item = itemStore.get(uid);
			if (item != null) {
				entryStore.updateAccountType(item, accountType);
				cache.invalidate(uid);
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
