/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2018
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.hollow.datamodel;

import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import com.netflix.hollow.api.objects.HollowList;
import com.netflix.hollow.core.write.objectmapper.HollowInline;
import com.netflix.hollow.core.write.objectmapper.HollowPrimaryKey;

import net.bluemind.directory.hollow.datamodel.consumer.HString;

@HollowPrimaryKey(fields = { "uid" })
public class AddressBookRecord extends AddressBookRecordIndexOnly {

	/**
	 * domain
	 */
	public String domain;
	/**
	 * created
	 */
	public Date created;
	/**
	 * updated
	 */
	public Date updated;
	/**
	 * email
	 */
	@HollowInline
	public String email;

	/**
	 * displayName
	 */
	@HollowInline
	public String name;

	/**
	 * PidTagSurname
	 */
	@HollowInline
	public String surname;
	/**
	 * PidTagGivenName
	 */
	@HollowInline
	public String givenName;
	/**
	 * PidTagTitle
	 */
	public String title;
	/**
	 * PidTagOfficeLocation
	 */
	public String officeLocation;
	/**
	 * PidTagDepartmentName
	 */
	public String departmentName;
	/**
	 * PidTagCompanyName
	 */
	public String companyName;
	/**
	 * PidTagAssistant
	 */
	public String assistant;
	public String addressBookManagerDistinguishedName;
	/**
	 * PidTagAddressBookPhoneticGivenName
	 */
	public String addressBookPhoneticGivenName;
	/**
	 * PidTagAddressBookPhoneticSurname
	 */
	public String addressBookPhoneticSurname;
	public String addressBookPhoneticCompanyName;
	public String addressBookPhoneticDepartmentName;
	/**
	 * PidTagStreetAddress
	 */
	public String streetAddress;
	public String postOfficeBox;
	/**
	 * PidTagLocality
	 */
	public String locality;
	/**
	 * PidTagStateOrProvince
	 */
	public String stateOrProvince;
	/**
	 * PidTagPostalcode
	 */
	public String postalCode;
	/**
	 * PidTagCountry
	 */
	public String country;
	/**
	 * Datalocation
	 */
	public DataLocation dataLocation;
	/**
	 * PidTagBusinessTelephoneNumber
	 */
	public String businessTelephoneNumber;
	/**
	 * PidTagHomeTelephoneNumber
	 */
	public String homeTelephoneNumber;
	/**
	 * PidTagBusiness2TelephoneNumbers
	 */
	public String business2TelephoneNumbers;
	/**
	 * PidTagBusiness2TelephoneNumbers
	 */
	public String home2TelephoneNumber;
	/**
	 * PidTagMobileTelephoneNumber
	 */
	public String mobileTelephoneNumber;
	/**
	 * PidTagPagerTelephoneNumber
	 */
	public String pagerTelephoneNumber;
	/**
	 * PidTagPrimaryFaxNumber
	 */
	public String primaryFaxNumber;
	/**
	 * PidTagAssistantTelephoneNumber
	 */
	public String assistantTelephoneNumber;
	/**
	 * PidTagUserCertificate
	 */
	public String userCertificate;
	/**
	 * PidTagAddressBookX509Certificate
	 */
	public List<Cert> addressBookX509Certificate;
	/**
	 * PidTagUserX509Certificate
	 */
	public List<Cert> userX509Certificate;
	/**
	 * PidTagThumbnailPhoto
	 */
	public byte[] thumbnail;

	public boolean hidden;

	private static <V, R> R ifNotNul(V source, Function<V, R> map) {
		return Optional.ofNullable(source).map(map).orElse(null);
	}

	private static String getValue(HString source) {
		return ifNotNul(source, HString::getValue);
	}

	private static Date getValue(net.bluemind.directory.hollow.datamodel.consumer.Date source) {
		return ifNotNul(source, s -> new Date(s.getValue()));
	}

	private static <T1, T2> List<T2> getValue(HollowList<T1> source, Function<T1, T2> map) {
		return ifNotNul(source, s -> s.stream().map(map).toList());
	}

	public AddressBookRecord(net.bluemind.directory.hollow.datamodel.consumer.AddressBookRecord source) {
		this(source.getUid(), source.getDistinguishedName(), getValue(source.getDomain()), getValue(source.getKind()),

				source.getEmails().stream().map(Email::new).toList(),

				getValue(source.getCreated()), getValue(source.getUpdated()), source.getEmail(), source.getMinimalid(),
				source.getName(), source.getSurname(), source.getGivenName(),

				getValue(source.getTitle()), getValue(source.getOfficeLocation()), getValue(source.getDepartmentName()),
				getValue(source.getCompanyName()), getValue(source.getAssistant()),

				getValue(source.getAddressBookManagerDistinguishedName()),
				getValue(source.getAddressBookPhoneticGivenName()), getValue(source.getAddressBookPhoneticSurname()),
				getValue(source.getAddressBookPhoneticCompanyName()),
				getValue(source.getAddressBookPhoneticDepartmentName()),

				getValue(source.getStreetAddress()), getValue(source.getPostOfficeBox()),
				getValue(source.getLocality()), getValue(source.getStateOrProvince()), getValue(source.getPostalCode()),
				getValue(source.getCountry()),

				Optional.ofNullable(source.getDataLocation()).map(DataLocation::new).orElse(null),

				getValue(source.getBusinessTelephoneNumber()), getValue(source.getHomeTelephoneNumber()),
				getValue(source.getBusiness2TelephoneNumbers()), getValue(source.getHome2TelephoneNumber()),
				getValue(source.getMobileTelephoneNumber()), getValue(source.getPagerTelephoneNumber()),
				getValue(source.getPrimaryFaxNumber()), getValue(source.getAssistantTelephoneNumber()),
				getValue(source.getUserCertificate()),

				getValue(source.getAddressBookX509Certificate(), Cert::new),
				getValue(source.getUserX509Certificate(), Cert::new),

				source.getThumbnail(), source.getHidden(), getValue(source.getAnr(), AnrToken::new));
	}

	public AddressBookRecord(String uid, String distinguishedName, String domain, String kind,

			List<Email> emails,

			Date created, Date updated, String email, int minimalid, String name, String surname, String givenName,

			String title, String officeLocation, String departmentName, String companyName, String assistant,

			String addressBookManagerDistinguishedName, String addressBookPhoneticGivenName,
			String addressBookPhoneticSurname, String addressBookPhoneticCompanyName,
			String addressBookPhoneticDepartmentName,

			String streetAddress, String postOfficeBox, String locality, String stateOrProvince, String postalCode,
			String country,

			DataLocation dataLocation,

			String businessTelephoneNumber, String homeTelephoneNumber, String business2TelephoneNumbers,
			String home2TelephoneNumber, String mobileTelephoneNumber, String pagerTelephoneNumber,
			String primaryFaxNumber, String assistantTelephoneNumber, String userCertificate,

			List<Cert> addressBookX509Certificate, List<Cert> userX509Certificate,

			byte[] thumbnail, boolean hidden, List<AnrToken> anr) {
		super();
		this.uid = uid;
		this.distinguishedName = distinguishedName;
		this.domain = domain;
		this.kind = kind;
		this.emails = emails;
		this.created = created;
		this.updated = updated;
		this.email = email;
		this.minimalid = minimalid;
		this.name = name;
		this.surname = surname;
		this.givenName = givenName;
		this.title = title;
		this.officeLocation = officeLocation;
		this.departmentName = departmentName;
		this.companyName = companyName;
		this.assistant = assistant;
		this.addressBookManagerDistinguishedName = addressBookManagerDistinguishedName;
		this.addressBookPhoneticGivenName = addressBookPhoneticGivenName;
		this.addressBookPhoneticSurname = addressBookPhoneticSurname;
		this.addressBookPhoneticCompanyName = addressBookPhoneticCompanyName;
		this.addressBookPhoneticDepartmentName = addressBookPhoneticDepartmentName;
		this.streetAddress = streetAddress;
		this.postOfficeBox = postOfficeBox;
		this.locality = locality;
		this.stateOrProvince = stateOrProvince;
		this.postalCode = postalCode;
		this.country = country;
		this.dataLocation = dataLocation;
		this.businessTelephoneNumber = businessTelephoneNumber;
		this.homeTelephoneNumber = homeTelephoneNumber;
		this.business2TelephoneNumbers = business2TelephoneNumbers;
		this.home2TelephoneNumber = home2TelephoneNumber;
		this.mobileTelephoneNumber = mobileTelephoneNumber;
		this.pagerTelephoneNumber = pagerTelephoneNumber;
		this.primaryFaxNumber = primaryFaxNumber;
		this.assistantTelephoneNumber = assistantTelephoneNumber;
		this.userCertificate = userCertificate;
		this.addressBookX509Certificate = addressBookX509Certificate;
		this.userX509Certificate = userX509Certificate;
		this.thumbnail = thumbnail;
		this.hidden = hidden;
		this.anr = anr;
	}

	public AddressBookRecord() {
	}

	public String getDomain() {
		return domain;
	}

	public List<Email> getEmails() {
		return emails;
	}

	public Date getCreated() {
		return created;
	}

	public Date getUpdated() {
		return updated;
	}

	public String getEmail() {
		return email;
	}

	public String getName() {
		return name;
	}

	public String getSurname() {
		return surname;
	}

	public String getGivenName() {
		return givenName;
	}

	public String getTitle() {
		return title;
	}

	public String getOfficeLocation() {
		return officeLocation;
	}

	public String getDepartmentName() {
		return departmentName;
	}

	public String getCompanyName() {
		return companyName;
	}

	public String getAssistant() {
		return assistant;
	}

	public String getAddressBookManagerDistinguishedName() {
		return addressBookManagerDistinguishedName;
	}

	public String getAddressBookPhoneticGivenName() {
		return addressBookPhoneticGivenName;
	}

	public String getAddressBookPhoneticSurname() {
		return addressBookPhoneticSurname;
	}

	public String getAddressBookPhoneticCompanyName() {
		return addressBookPhoneticCompanyName;
	}

	public String getAddressBookPhoneticDepartmentName() {
		return addressBookPhoneticDepartmentName;
	}

	public String getStreetAddress() {
		return streetAddress;
	}

	public String getPostOfficeBox() {
		return postOfficeBox;
	}

	public String getLocality() {
		return locality;
	}

	public String getStateOrProvince() {
		return stateOrProvince;
	}

	public String getPostalCode() {
		return postalCode;
	}

	public String getCountry() {
		return country;
	}

	public DataLocation getDataLocation() {
		return dataLocation;
	}

	public String getBusinessTelephoneNumber() {
		return businessTelephoneNumber;
	}

	public String getHomeTelephoneNumber() {
		return homeTelephoneNumber;
	}

	public String getBusiness2TelephoneNumbers() {
		return business2TelephoneNumbers;
	}

	public String getHome2TelephoneNumber() {
		return home2TelephoneNumber;
	}

	public String getMobileTelephoneNumber() {
		return mobileTelephoneNumber;
	}

	public String getPagerTelephoneNumber() {
		return pagerTelephoneNumber;
	}

	public String getPrimaryFaxNumber() {
		return primaryFaxNumber;
	}

	public String getAssistantTelephoneNumber() {
		return assistantTelephoneNumber;
	}

	public String getUserCertificate() {
		return userCertificate;
	}

	public List<Cert> getAddressBookX509Certificate() {
		return addressBookX509Certificate;
	}

	public List<Cert> getUserX509Certificate() {
		return userX509Certificate;
	}

	public byte[] getThumbnail() {
		return thumbnail;
	}

	public boolean getHidden() {
		return hidden;
	}

	public List<AnrToken> getAnr() {
		return anr;
	}

}
