/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.hollow.datamodel.producer;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.IAddressBooksMgmt;
import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.ICalendarsMgmt;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.hollow.datamodel.producer.Value.BooleanValue;
import net.bluemind.directory.hollow.datamodel.producer.Value.ByteArrayValue;
import net.bluemind.directory.hollow.datamodel.producer.Value.DateValue;
import net.bluemind.directory.hollow.datamodel.producer.Value.ListValue;
import net.bluemind.directory.hollow.datamodel.producer.Value.NullValue;
import net.bluemind.directory.hollow.datamodel.producer.Value.StringValue;
import net.bluemind.directory.hollow.datamodel.utils.JPEGThumbnail;
import net.bluemind.externaluser.api.ExternalUser;
import net.bluemind.externaluser.api.IExternalUser;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.IGroup;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailshare.api.IMailshare;
import net.bluemind.mailshare.api.Mailshare;
import net.bluemind.resource.api.IResources;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.Server;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public abstract class DirEntrySerializer {

	public enum Property {
		Email, SmtpAddress, DisplayName, Account, Surname, GivenName, AddressBookProxyAddresses, OfficeLocation,
		DisplayType, ObjectType, SendRichInfo, BusinessTelephoneNumber, StreetAddress, Locality, StateOrProvince,
		PostalCode, Country, Title, CompanyName, Assistant, DepartmentName, AddressBookTargetAddress,
		HomeTelephoneNumber, Business2TelephoneNumbers, PrimaryFaxNumber, MobileTelephoneNumber,
		AssistantTelephoneNumber, PagerTelephoneNumber, Comment, UserCertificate, UserX509Certificate,
		AddressBookX509Certificate, AddressBookHomeMessageDatabaseAscii, AddressBookDisplayNamePrintableAscii,
		ThumbnailPhoto, postOfficeBox, AddressBookManagerDistinguishedName, Kind, DataLocation, Created, Updated, Hidden

	}

	private static final Logger logger = LoggerFactory.getLogger(DirEntrySerializer.class);

	protected final ItemValue<DirEntry> dirEntry;
	protected final String domainUid;

	protected DirEntrySerializer(ItemValue<DirEntry> dirEntry, String domainUid) {
		this.dirEntry = dirEntry;
		this.domainUid = domainUid;
	}

	public ItemValue<DirEntry> getDirEntry() {
		return dirEntry;
	}

	public static List<? extends DirEntrySerializer> get(String domainUid, BaseDirEntry.Kind kind,
			Collection<ItemValue<DirEntry>> dirEntries) {
		Map<Object, ItemValue<DirEntry>> dirEntriesByUid = dirEntries.stream()
				.collect(Collectors.toMap(entry -> entry.uid, Function.identity()));
		switch (kind) {
		case USER:
			List<String> userUids = dirEntries.stream().map(entry -> entry.uid).toList();
			List<ItemValue<User>> users = provider().instance(IUser.class, domainUid) //
					.getMultipleVcardOnly(userUids);
			return users.stream().map(user -> new UserSerializer(user, dirEntriesByUid.get(user.uid), domainUid))
					.toList();

		case GROUP:
			return dirEntries.stream().map(dirEntry -> {
				ItemValue<Group> group = provider().instance(IGroup.class, domainUid).getComplete(dirEntry.uid);
				return group != null ? new GroupSerializer(group, dirEntry, domainUid) : null;
			}).filter(o -> o != null).toList();

		case RESOURCE:
			return dirEntries.stream().map(dirEntry -> {
				ResourceDescriptor resource = provider().instance(IResources.class, domainUid).get(dirEntry.uid);
				return resource != null ? new ResourceSerializer(resource, dirEntry, domainUid) : null;
			}).filter(o -> o != null).toList();

		case MAILSHARE:
			return dirEntries.stream().map(dirEntry -> {
				ItemValue<Mailshare> mailshare = provider().instance(IMailshare.class, domainUid)
						.getComplete(dirEntry.uid);
				return mailshare != null ? new MailshareSerializer(mailshare, dirEntry, domainUid) : null;
			}).filter(o -> o != null).toList();

		case CALENDAR:
			return dirEntries.stream().map(dirEntry -> {
				CalendarDescriptor calendar = provider().instance(ICalendarsMgmt.class).getComplete(dirEntry.uid);
				return calendar != null ? new CalendarSerializer(calendar, dirEntry, domainUid) : null;
			}).filter(o -> o != null).toList();

		case ADDRESSBOOK:
			return dirEntries.stream().map(dirEntry -> {
				AddressBookDescriptor addressbook = provider().instance(IAddressBooksMgmt.class)
						.getComplete(dirEntry.uid);
				return addressbook != null ? new DomainAddressBookSerializer(addressbook, dirEntry, domainUid) : null;
			}).filter(o -> o != null).toList();

		case EXTERNALUSER:
			return dirEntries.stream().map(dirEntry -> {

				ItemValue<ExternalUser> externalUser = provider().instance(IExternalUser.class, domainUid)
						.getComplete(dirEntry.uid);
				if (externalUser != null) {
					ItemValue<Mailbox> mbox = provider().instance(IMailboxes.class, domainUid)
							.byEmail(dirEntry.value.email);
					logger.info("External user found {},  mailbox exists? {}", dirEntry.value.email, mbox != null);
					if (mbox == null) {
						return new ExternalUserSerializer(externalUser, dirEntry, domainUid);
					} else {
						logger.warn("Skip external user {}. Email conflicts with internal dirEntry {}", dirEntry.uid,
								dirEntry.value.email);
					}
				}
				return null;
			}).filter(o -> o != null).toList();
		default:
			throw new IllegalArgumentException("DirEntry kind " + kind + " not supported");
		}

	}

	public static Optional<DirEntrySerializer> get(String domainUid, ItemValue<DirEntry> dirEntry) {
		DirEntrySerializer ret = null;
		switch (dirEntry.value.kind) {
		case USER:
			ItemValue<User> user = provider().instance(IUser.class, domainUid).getComplete(dirEntry.uid);
			if (user != null) {
				ret = new UserSerializer(user, dirEntry, domainUid);
			}
			break;
		case GROUP:
			ItemValue<Group> group = provider().instance(IGroup.class, domainUid).getComplete(dirEntry.uid);
			if (group != null) {
				ret = new GroupSerializer(group, dirEntry, domainUid);
			}
			break;
		case RESOURCE:
			ResourceDescriptor resource = provider().instance(IResources.class, domainUid).get(dirEntry.uid);
			if (resource != null) {
				ret = new ResourceSerializer(resource, dirEntry, domainUid);
			}
			break;
		case MAILSHARE:
			ItemValue<Mailshare> mailshare = provider().instance(IMailshare.class, domainUid).getComplete(dirEntry.uid);
			if (mailshare != null) {
				ret = new MailshareSerializer(mailshare, dirEntry, domainUid);
			}
			break;
		case CALENDAR:
			CalendarDescriptor calendar = provider().instance(ICalendarsMgmt.class).getComplete(dirEntry.uid);
			if (calendar != null) {
				ret = new CalendarSerializer(calendar, dirEntry, domainUid);
			}
			break;
		case ADDRESSBOOK:
			AddressBookDescriptor addressbook = provider().instance(IAddressBooksMgmt.class).getComplete(dirEntry.uid);
			if (addressbook != null) {
				ret = new DomainAddressBookSerializer(addressbook, dirEntry, domainUid);
			}
			break;
		case EXTERNALUSER:
			ItemValue<ExternalUser> externalUser = provider().instance(IExternalUser.class, domainUid)
					.getComplete(dirEntry.uid);
			if (externalUser != null) {
				ItemValue<Mailbox> mbox = provider().instance(IMailboxes.class, domainUid)
						.byEmail(dirEntry.value.email);
				logger.info("External user found {},  mailbox exists? {}", dirEntry.value.email, mbox != null);
				if (mbox == null) {
					ret = new ExternalUserSerializer(externalUser, dirEntry, domainUid);
				} else {
					logger.warn("Skip external user {}. Email conflicts with internal dirEntry {}", dirEntry.uid,
							dirEntry.value.email);
				}
			}
			break;
		default:
			throw new IllegalArgumentException("DirEntry kind " + dirEntry.value.kind + " not supported");
		}
		return Optional.ofNullable(ret);
	}

	private static IServiceProvider provider() {
		return ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
	}

	public Value get(Property property) {
		switch (property) {
		case Created:
			return new DateValue(dirEntry.created);
		case Updated:
			return dirEntry.updated != null ? new DateValue(dirEntry.updated) : new NullValue();
		case Email:
			return getEmailAddress(dirEntry.value.displayName);
		case AddressBookManagerDistinguishedName:
			return new StringValue("/");
		case ThumbnailPhoto:
			return getPhoto();
		case Kind:
			return new StringValue(dirEntry.value.kind.name());
		case DataLocation:
			return new ListValue(getDataLocation(dirEntry));
		case Hidden:
			return new BooleanValue(dirEntry.value.hidden);
		default:
			return Value.NULL;
		}
	}

	private List<?> getDataLocation(ItemValue<DirEntry> dirEntry) {
		String server = dirEntry.value.dataLocation;
		if (null != server) {
			ItemValue<Server> srv = provider().instance(IServer.class, "default").getComplete(server);
			if (srv != null) {
				String address = srv.value.address();
				return Arrays.asList(address, server);
			}
		}
		return Collections.emptyList();
	}

	private Value getPhoto() {
		try {
			IDirectory directory = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IDirectory.class, domainUid);
			byte[] pngPhoto = directory.getEntryPhoto(dirEntry.value.entryUid);
			if (pngPhoto != null && pngPhoto.length != 0) {
				return new ByteArrayValue(JPEGThumbnail.scaleImage(pngPhoto, 120, 120));
			}
		} catch (Exception e) {
			// no pic
		}
		return Value.NULL;

	}

	protected Value getEmailAddress(String name) {
		if (dirEntry.value.email != null) {
			return new StringValue(dirEntry.value.email);
		}
		if (name != null) {
			return new StringValue(name + "@" + domainUid);
		}

		return Value.NULL;
	}

}
