/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2018
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.hollow.datamodel.producer;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.hollow.api.consumer.HollowConsumer;
import com.netflix.hollow.api.consumer.HollowConsumer.AnnouncementWatcher;
import com.netflix.hollow.api.consumer.HollowConsumer.BlobRetriever;
import com.netflix.hollow.api.consumer.fs.HollowFilesystemAnnouncementWatcher;
import com.netflix.hollow.api.consumer.fs.HollowFilesystemBlobRetriever;
import com.netflix.hollow.api.producer.HollowProducer;
import com.netflix.hollow.api.producer.HollowProducer.BlobStorageCleaner;
import com.netflix.hollow.api.producer.HollowProducer.Incremental;
import com.netflix.hollow.api.producer.fs.HollowFilesystemAnnouncer;
import com.netflix.hollow.api.producer.fs.HollowFilesystemPublisher;
import com.netflix.hollow.core.write.objectmapper.RecordPrimaryKey;

import net.bluemind.common.hollow.BmFilesystemBlobStorageCleaner;
import net.bluemind.config.InstallationId;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.serialization.HzHollowAnnouncer;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.DataLocation;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.consumer.DirectoryVersionReader;
import net.bluemind.directory.hollow.datamodel.producer.impl.DomainVersions;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;

public class DirectorySerializer implements IDirectorySerializer {

	private HollowConsumer.BlobRetriever blobRetriever;

	private HollowConsumer.AnnouncementWatcher announcementWatcher;
	private static final String BASE_DATA_DIR = "/var/spool/bm-hollowed/directory";

	private static final Logger logger = LoggerFactory.getLogger(DirectorySerializer.class);

	private ServerSideServiceProvider prov;
	private Incremental producer;
	private final String domainUid;
	private final Object produceLock;
	private final Object rebuildLock;

	private static final boolean DROP_HIDDEN = new File("/etc/bm/hollow.no.hidden").exists();

	public DirectorySerializer(String domainUid) {
		this.domainUid = domainUid;
		this.produceLock = new Object();
		this.rebuildLock = new Object();
		initOrReset();
	}

	public void start() {
		if (!restoreIfAvailable(producer, blobRetriever, announcementWatcher)) {
			produce();
		}
	}

	private void initOrReset() {
		try {
			init();
		} catch (HollowCorruptedException e) {
			logger.warn("Trying to recreate from scratch, cause: {}", e.getMessage());
			remove();
			init();
		}
	}

	public void init() {
		synchronized (produceLock) {
			this.prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

			File localPublishDir = createDataDir();

			HollowFilesystemPublisher publisher = new HollowFilesystemPublisher(localPublishDir.toPath());
			HollowFilesystemAnnouncer announcer = new HzHollowAnnouncer("directory/" + domainUid, localPublishDir);
			try {
				this.announcementWatcher = new HollowFilesystemAnnouncementWatcher(localPublishDir.toPath());
			} catch (NumberFormatException nfe) {
				throw new HollowCorruptedException("Corrupted hollow directory, invalid announced.version format", nfe);
			}

			BlobStorageCleaner cleaner = new BmFilesystemBlobStorageCleaner(localPublishDir, 10);
			this.producer = HollowProducer.withPublisher(publisher).withAnnouncer(announcer) //
					.noIntegrityCheck().withBlobStorageCleaner(cleaner).buildIncremental();
			producer.initializeDataModel(AddressBookRecord.class);
			producer.initializeDataModel(OfflineAddressBook.class);
			logger.info("Announcement watcher current version: {}", announcementWatcher.getLatestVersion());
			this.blobRetriever = new HollowFilesystemBlobRetriever(localPublishDir.toPath());
		}
	}

	/**
	 * @return the version of the hollow snap
	 */
	public long produce() {
		synchronized (produceLock) {
			return serializeIncrement();
		}
	}

	private File createDataDir() {
		File localPublishDir = getDataDir();
		localPublishDir.mkdirs();
		return localPublishDir;
	}

	public File getDataDir() {
		return new File(BASE_DATA_DIR, domainUid);
	}

	private boolean restoreIfAvailable(Incremental producer, BlobRetriever retriever,
			AnnouncementWatcher unpinnableAnnouncementWatcher) {

		long latestVersion = unpinnableAnnouncementWatcher.getLatestVersion();
		try {
			if (latestVersion != AnnouncementWatcher.NO_ANNOUNCEMENT_AVAILABLE) {
				producer.restore(latestVersion, retriever);
				return true;
			}
			return false;
		} catch (Exception e) {
			logger.error("Could not restore existing hollow snapshot for {}", domainUid, e);
			return false;
		}
	}

	private long serializeIncrement() {
		Map<String, OfflineAddressBook> oabs = new HashMap<>();

		IInCoreDomains domApi = prov.instance(IInCoreDomains.class);
		IDirectory dirApi = prov.instance(IDirectory.class, domainUid);
		IMailboxes mboxApi = prov.instance(IMailboxes.class, domainUid);
		ItemValue<Domain> domain = domApi.getUnfiltered(domainUid);
		String installationId = InstallationId.getIdentifier();

		long version = Optional.ofNullable(DomainVersions.get().getIfPresent(domainUid)).orElse(0L);

		if (version == 0l) {
			DirectoryVersionReader reader = new DirectoryVersionReader(domainUid);
			version = reader.version();
			logger.info("[{}] Version fetched from hollow root is {}", domainUid, version);
		}

		ContainerChangeset<String> changeset = dirApi.changeset(version);
		List<String> allUids = new ArrayList<>(Sets.newHashSet(Iterables.concat(changeset.created, changeset.updated)));
		logger.info("Sync from v{} gave +{} / -{} uid(s)", version, allUids.size(), changeset.deleted.size());
		final Map<String, DataLocation> locationCache = new HashMap<>();

		long hollowVersion = producer.runIncrementalCycle(populator -> {
			OfflineAddressBook oab = oabs.computeIfAbsent(domainUid,
					d -> EntryToAdressBookMapper.createOabEntry(domain, changeset.version));
			oab.sequence = (int) changeset.version;
			populator.addOrModify(oab);
			for (List<String> dirPartition : Lists.partition(allUids, 100)) {
				List<ItemValue<DirEntry>> entries = loadEntries(dirApi, dirPartition);
				List<String> uidWithEmails = entries.stream().filter(iv -> iv.value.email != null).map(iv -> iv.uid)
						.toList();

				List<ItemValue<Mailbox>> mailboxes = mboxApi.multipleGet(uidWithEmails);
				for (ItemValue<DirEntry> entry : entries) {
					ItemValue<Mailbox> mailbox = mailboxes.stream().filter(m -> m.uid.equals(entry.value.entryUid))
							.findAny().orElse(null);
					EntryToAdressBookMapper.map(domain, entry, mailbox, locationCache, installationId)
							.ifPresent(rec -> {
								if (dropHiddenEntry(entry)) {
									populator.delete(new RecordPrimaryKey("AddressBookRecord",
											new String[] { entry.value.entryUid }));
								} else {
									populator.addOrModify(rec);
								}

							});
				}
			}
			for (String uidToRm : changeset.deleted) {
				populator.delete(new RecordPrimaryKey("AddressBookRecord", new String[] { uidToRm }));
			}
		});

		logger.info("Created new incremental hollow snap (dir v{}, hollow v{})", changeset.version, hollowVersion);
		DomainVersions.get().put(domainUid, changeset.version);
		return hollowVersion;
	}

	private boolean dropHiddenEntry(ItemValue<DirEntry> de) {
		return DROP_HIDDEN && de.value.hidden;
	}

	private List<ItemValue<DirEntry>> loadEntries(IDirectory dirApi, List<String> dirPartition) {
		List<ItemValue<DirEntry>> entries;
		try {
			entries = dirApi.getMultiple(dirPartition);
		} catch (ServerFault e) {
			entries = new ArrayList<>();
			for (String uid : dirPartition) {
				try {
					entries.add(dirApi.getMultiple(Arrays.asList(uid)).get(0));
				} catch (ServerFault e1) {
					logger.warn("Skipping broken item {}", uid, e1);
				}
			}
		}

		return entries.stream().filter(this::supportedType).collect(Collectors.toList());
	}

	public void remove() {
		synchronized (produceLock) {
			try {
				logger.info("Removing data dir {}", getDataDir());
				deleteDataDir();
				DomainVersions.get().invalidate(domainUid);
			} catch (Exception e) {
				logger.warn("Cannot delete data dir {}", getDataDir());
			}
		}
	}

	public void rebuild() {
		synchronized (rebuildLock) {
			remove();
			init();
			produce();
		}
	}

	private void deleteDataDir() throws IOException {
		Path directory = Paths.get(getDataDir().getAbsolutePath());
		Files.walkFileTree(directory, new SimpleFileVisitor<Path>() {
			@Override
			public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
				Files.delete(file);
				return FileVisitResult.CONTINUE;
			}

			@Override
			public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
				Files.delete(dir);
				return FileVisitResult.CONTINUE;
			}
		});

	}

	@Override
	public HollowConsumer.BlobRetriever getBlobRetriever() {
		return blobRetriever;
	}

	@Override
	public long getLastVersion() {
		return announcementWatcher.getLatestVersion();
	}

	@SuppressWarnings("serial")
	private class HollowCorruptedException extends RuntimeException {

		public HollowCorruptedException(String message, Throwable cause) {
			super(message, cause);
		}

	}
}
