<?php

/* BEGIN LICENSE
 * Copyright Â© Blue Mind SAS, 2012-2017
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
namespace BM;
require_once 'GlobalEnv.php';

/**
 * Implementation of net.bluemind.directory.api.IDirectory.
 */

class DirectoryClient {

  protected $base;
  protected $sid;
  protected $domain;

  /*
   * Constructor.
   *
   * @param base
   * @param sid
   * @param domain
   *
   */
  public function __construct($base, $sid , $domain) {
    $this->sid = $sid;
    $this->base = $base."/api/directory/{domain}";
    $this->domain = $domain;
    $this->base = str_replace("{domain}", urlencode($domain), $this->base);
  }


  /*
   * @param since
   * @return
   */
  public function changeset( $since  ) {
    $postUri = "/_changeset";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array(
      "since" => $since);

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param path
   * @return
   */
  public function delete_( $path  ) {
    $postUri = "/{path}";
    $method = "DELETE";

    $postUri = str_replace("{path}", urlencode($path), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function deleteByEntryUid( $entryUid  ) {
    $postUri = "/_byentryuid/{entryUid}";
    $method = "DELETE";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function findByEntryUid( $entryUid  ) {
    $postUri = "/entry-uid/{entryUid}";
    $method = "GET";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function findItemValueByEntryUid( $entryUid  ) {
    $postUri = "/item-value-uid/{entryUid}";
    $method = "GET";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param email
   * @return
   */
  public function getByEmail( $email  ) {
    $postUri = "/_byEmail/{email}";
    $method = "GET";

    $postUri = str_replace("{email}", urlencode($email), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param roles
   * @return
   */
  public function getByRoles( $roles  ) {
    $postUri = "/_byroles";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $roles;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param path
   * @return
   */
  public function getEntries( $path  ) {
    $postUri = "/_childs";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $path;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param path
   * @return
   */
  public function getEntry( $path  ) {
    $postUri = "/_entry";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $path;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function getEntryIcon( $entryUid  ) {
    $postUri = "/entry-uid/{entryUid}/icon";
    $method = "GET";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function getEntryPhoto( $entryUid  ) {
    $postUri = "/entry-uid/{entryUid}/photo";
    $method = "GET";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param path
   * @return
   */
  public function getIcon( $path  ) {
    $postUri = "/_icon/{path}";
    $method = "GET";

    $postUri = str_replace("{path}", urlencode($path), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param id
   * @return
   */
  public function getMultiple( $id  ) {
    $postUri = "/_mget";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $id;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param id
   * @return
   */
  public function getMultipleLight( $id  ) {
    $postUri = "/_mgetlight";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $id;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param entryUid
   * @return
   */
  public function getRolesForDirEntry( $entryUid  ) {
    $postUri = "/entry-uid/{entryUid}/rolesfor_";
    $method = "GET";

    $postUri = str_replace("{entryUid}", urlencode($entryUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param ouUid
   * @return
   */
  public function getRolesForOrgUnit( $ouUid  ) {
    $postUri = "/ou-uid/{ouUid}/rolesfor_";
    $method = "GET";

    $postUri = str_replace("{ouUid}", urlencode($ouUid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @return
   */
  public function getRoot() {
    $postUri = "";
    $method = "GET";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param uid
   * @return
   */
  public function getVCard( $uid  ) {
    $postUri = "/{uid}/_vcard";
    $method = "GET";

    $postUri = str_replace("{uid}", urlencode($uid), $postUri);

    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param query
   * @return
   */
  public function search( $query  ) {
    $postUri = "/_search";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $query;
    return $this->execute($url, $queryParam, $body, $method);
  }

  /*
   * @param query
   * @return
   */
  public function searchUids( $query  ) {
    $postUri = "/_searchUids";
    $method = "POST";


    $url = $this->base.$postUri;

    $queryParam = array();

    $body = null;
    $body = $query;
    return $this->execute($url, $queryParam, $body, $method);
  }


  /*
   * Execute the request
   *
   * @param url
   * @param data
   * @param body
   */
  private function execute($url, $queryParam, $body, $method) {

    $curl = curl_init();

    $headers = array();
    array_push($headers, 'X-BM-ApiKey: '.$this->sid);

    if (sizeof($queryParam) > 0) {
      $url .= '?'.http_build_query($queryParam);
    }

    curl_setopt_array($curl, GlobalEnv::$curlOptions + array(
      CURLOPT_URL => $url,
      CURLOPT_HTTPHEADER => $headers,
      CURLOPT_CUSTOMREQUEST => $method)
    );

    if ($method == 'POST') {
      curl_setopt($curl, CURLOPT_POST, TRUE);
      curl_setopt($curl, CURLOPT_POSTFIELDS, array());
    }

    if (is_resource($body)) {
      if ($method == 'PUT') {
        curl_setopt($curl, CURLOPT_PUT, TRUE);
      }
      $size = fstat($body)['size'];
      curl_setopt($curl, CURLOPT_INFILE, $body);
      curl_setopt($curl, CURLOPT_INFILESIZE, $size);
    } else {
      if (is_object($body) && method_exists($body, 'serialize')) {
        $body = $body->serialize();
      } else if (is_object($body)) {
        $body = json_encode($body);
      } else if (is_array($body)) {
        $body = json_encode($body);
      } else if (is_string($body)) {
        $body = json_encode($body);
      }
      $size = strlen($body);
      array_push($headers, 'Content-Type: application/json');
      array_push($headers, 'Content-Length: '.$size);
      curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
    }

    $resp = curl_exec($curl);
    if ($resp === false) {
      throw new \Exception(curl_error($curl));
    } 
    curl_close($curl);
    if (!$resp) {
      return;
    }
    $js = json_decode($resp);
    if ($js === NULL) {
      return $resp;
    }
    if (isset($js->errorCode)) {
      throw new \Exception($js->errorCode . ': ' . (isset($js->message) ? ' : ' . $js->message : ''));
    }
    return $js;
  }

}
