/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.delivery.conversationreference.service;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.delivery.conversationreference.api.IConversationReference;
import net.bluemind.delivery.conversationreference.repository.ConversationReference;
import net.bluemind.delivery.conversationreference.repository.IConversationReferenceStore;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.repository.provider.RepositoryProvider;

public class ConversationReferenceService implements IConversationReference {

	private final IConversationReferenceStore store;
	private final long mailboxId;

	public ConversationReferenceService(BmContext context, ItemValue<Mailbox> mailbox) {
		mailboxId = mailbox.internalId;
		store = RepositoryProvider.instance(IConversationReferenceStore.class, context);
	}

	private ConversationReference create(Long messageIdHash) throws SQLException {
		var cr = ConversationReference.of(messageIdHash, messageIdHash, mailboxId);
		store.create(cr);
		return cr;
	}

	private boolean isValidMessageId(String msgid) {
		if (msgid == null) {
			return false;
		}
		if (msgid.isBlank() || msgid.isEmpty()) {
			return false;
		}
		// Don't consider invalid message_ids (<x@x.c> is the bare minimum)
		return msgid.startsWith("<") && msgid.endsWith(">") && msgid.length() > 6;
	}

	/*
	 * Scenarios:
	 * 
	 * unknown conversation, no references: create a new conversation with the
	 * message-id known conversation, new references: add a reference to message-id,
	 * add all references to the conversation.
	 * 
	 * Conflicting cases are handled by INSERT ON CONFLICT This means this scenario,
	 * received in order:
	 *
	 * MSG3: id: 300 references: 301
	 * 
	 * MSG1: id: 100 references: 200
	 * 
	 * MSG2: id: 200 references: 301
	 * 
	 * will produce 2 conflicting conversations for message id=301
	 */

	@Override
	public Long lookup(String messageId, Set<String> references) {
		HashFunction hf = Hashing.sipHash24();
		if (!isValidMessageId(messageId)) {
			// This should not happen, because postfix, in from of us should protect from
			// that, but it's not protected when injecting random emails from IMAP.
			messageId = "<" + UUID.randomUUID() + "@random-message-id.invalid>";
		}
		long messageIdHash = hf.hashBytes(messageId.getBytes()).asLong();
		long returnedConversationId;

		// Limit to 32 elements to avoid DoS
		List<Long> referencesHash = references.stream().filter(this::isValidMessageId)
				.map(s -> hf.hashBytes(s.getBytes()).asLong()).limit(32).collect(Collectors.toList());
		referencesHash.add(messageIdHash);
		try {
			returnedConversationId = store.get(mailboxId, referencesHash);
			if (returnedConversationId == 0L) {
				create(messageIdHash);
				returnedConversationId = messageIdHash;
			}
			// Store all known references
			if (returnedConversationId != 0L) {
				long cid = returnedConversationId; // Thank you java!
				store.create(referencesHash.stream().map(h -> ConversationReference.of(h, cid, mailboxId)).toList());
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		return returnedConversationId;
	}

	@Override
	public void restore(long convId, String messageId, Collection<String> r) {
		Collection<String> refs = r == null ? Collections.emptyList() : r;
		if (Strings.isNullOrEmpty(messageId) && refs.isEmpty()) {
			return;
		}
		HashFunction hf = Hashing.sipHash24();
		if (!isValidMessageId(messageId)) {
			// This should not happen, because postfix, in from of us should protect from
			// that, but it's not protected when injecting random emails from IMAP.
			messageId = "<" + UUID.randomUUID() + "@random-message-id.invalid>";
		}
		Set<String> mutableCopy = refs.stream().limit(31).collect(Collectors.toCollection(HashSet::new));
		mutableCopy.add(messageId);

		List<ConversationReference> referencesHash = mutableCopy.stream().filter(this::isValidMessageId)
				.map(s -> hf.hashBytes(s.getBytes()).asLong())
				.map(hash -> ConversationReference.of(hash, convId, mailboxId)).collect(Collectors.toList());
		try {
			store.create(referencesHash);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
