/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.webappdata.impl;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.dataprotect.service.tool.CoreCommonBackupWorker;
import net.bluemind.dataprotect.service.tool.ZipBuilder;
import net.bluemind.dataprotect.webappdata.impl.WebappDataBackupRepository.BackupWebAppDataRecord;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.server.api.Server;
import net.bluemind.webappdata.api.IWebAppData;
import net.bluemind.webappdata.api.IWebAppDataUids;

public class WebappDataBackupWorker implements IBackupWorker {
	private static Logger logger = LoggerFactory.getLogger(WebappDataBackupWorker.class);
	public static final Path WORKING_PATH = ZipBuilder.WORKING_PATH.resolve("webappdata");
	private static final String ZIP_FILE_PREFIX = IWebAppDataUids.TYPE;

	private static class WebappDataBackup extends CommonBackupWorker {
		private WebappDataBackupRepository repository;

		public WebappDataBackup(IDPContext ctx, Path p, PartGeneration partGen) {
			super(ctx, partGen);
			this.repository = new WebappDataBackupRepository(p);
		}

		@Override
		public List<MailboxIndexJson> partitionTreatment(String domainUid, String zipFile, List<String> partition) {
			String zipFileName = CommonBackupWorker.zipFilename(ZIP_FILE_PREFIX, zipFile);
			List<MailboxIndexJson> indexJson = new ArrayList<>();
			partition.forEach(uid -> {
				try {
					List<String> files = repository.writeDatas(backupWebAppDatas(uid));
					indexJson.add(new MailboxIndexJson(files, uid, domainUid, zipFileName));
				} catch (IOException e) {
					logError(e, "Unable to backup webappdatas {}", uid);
				}
			});
			indexJson.forEach(i -> i.zipFileName(zipFileName));
			indexJson.forEach(i -> i.domainUid(domainUid));
			return indexJson;
		}

		private MailboxIndexJsonList backupUsersWebappdata(ItemValue<Domain> domain) {
			var q = DirEntryQuery.allWithHidden();
			ListResult<String> searchUids = provider().instance(IDirectory.class, domain.uid).searchUids(q);
			return runIt(domain.uid, searchUids.values);
		}

		private List<BackupWebAppDataRecord> backupWebAppDatas(String ownerUid) {
			List<BackupWebAppDataRecord> datasToBackup = new ArrayList<>();
			provider().instance(IContainers.class) //
					.all(ContainerQuery.ownerAndType(ownerUid, IWebAppDataUids.TYPE)) //
					.forEach(cd -> {
						BackupWebAppDataRecord backupWebAppDatas = backupWebAppDatas(cd);
						if (backupWebAppDatas != null) {
							datasToBackup.add(backupWebAppDatas);
						}
					});
			return datasToBackup;
		}

		private BackupWebAppDataRecord backupWebAppDatas(ContainerDescriptor containerDescriptor) {
			logger.info("Backup webappdata {}", containerDescriptor.uid);
			IWebAppData webappDataApi = provider().instance(IWebAppData.class, containerDescriptor.uid);
			List<String> uids = webappDataApi.allUids();
			if (!uids.isEmpty()) {
				return new BackupWebAppDataRecord(containerDescriptor, webappDataApi.multipleGet(uids));
			} else {
				logger.debug("No webappdata for container {}", containerDescriptor.uid);
			}
			return null;
		}

		public void backupAll() {
			var domainsApi = provider().instance(IInCoreDomains.class);
			List<MailboxIndexJson> domainsZip = new ArrayList<>();
			domainsApi.allUnfiltered().stream().filter(d -> !"global.virt".equals(d.uid)).forEach(domain -> {
				logger.info("Backup webappdata for domain {}", domain.value.defaultAlias);
				domainsZip.addAll(backupUsersWebappdata(domain).mailboxesIndexInfo());
			});

			Path rootUserPath = repository.repositoryPath.resolveRootUserPath();
			try {
				RepositoryBackupPath.writeIndexFileToDir(rootUserPath, domainsZip);
			} catch (IOException e) {
				logError(e, "Error trying to write index.json {}", rootUserPath);
			}
		}
	}

	@Override
	public void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup) throws ServerFault {
		cleanup(ctx, partGen, null);
		Path outputFolder = WebappDataBackupRepository.DEFAULT_PATH;
		long startDate = Instant.now().getEpochSecond();

		try {
			if (!Files.exists(WORKING_PATH)) {
				Files.createDirectories(WORKING_PATH);
			}

			logger.info("Starting webappdatas backup in {}", WORKING_PATH);
			ctx.info("Starting webappdatas backup");
			new WebappDataBackup(ctx, WORKING_PATH, partGen).backupAll();
			CoreCommonBackupWorker.prepareDataDirs(ctx, toBackup, WORKING_PATH, outputFolder);
		} catch (IOException e) {
			partGen.withErrors = true;
			ctx.error(e, "Unable to create temporary directory for webappdata backup");
		} finally {
			ctx.info(String.format("Ending webappdatas backup in %d seconds",
					Instant.now().getEpochSecond() - startDate));
		}
	}

	@Override
	public String getDataType() {
		return WorkerDataType.WEBAPPDATA.value;
	}

	@Override
	public Set<String> getDataDirs() {
		return Set.of(WebappDataBackupRepository.DEFAULT_PATH.toString());
	}

	@Override
	public boolean supportsTag(String tag) {
		return CoreCommonBackupWorker.supportsTag(tag);
	}
}
