/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.todolist.impl;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.email.SendIcs;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;
import net.bluemind.dataprotect.todolist.impl.pg.SendUserTodolistsICSTasksPg;
import net.bluemind.user.api.User;

public class SendUserTodolistsICSTasks extends AbstractRestoreTodoLists {
	private static final Logger logger = LoggerFactory.getLogger(SendUserTodolistsICSTasks.class);

	private final RestoreActionExecutor<EmailData> executor;
	private final ResourceBundle bundle;

	@SuppressWarnings("unchecked")
	public SendUserTodolistsICSTasks(DataProtectGeneration backup, Restorable item,
			RestoreActionExecutor<? extends IRestoreActionData> executor) {
		super(backup, item);
		this.executor = (RestoreActionExecutor<EmailData>) executor;
		this.bundle = ResourceBundle.getBundle("OSGI-INF/l10n/RestoreTodo", Locale.of(ServerSideServiceProvider
				.getProvider(SecurityContext.SYSTEM).getContext().getSecurityContext().getLang()));
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startExportMsg(DPError.DPKind.TODOLISTS.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting export for uid {}", restorableItem.entryUid());
		}

		TodolistBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new SendUserTodolistsICSTasksPg(backup, restorableItem, executor).run();
		}

	}

	protected void restore(TodolistBackupRepository repository) {
		SendIcs sendEmail = new SendIcs(restorableItem, executor);

		try {
			List<String> todoContainerUids = repository.getRestorableTodolistUids(restorableItem.domain(),
					restorableItem.entryUid());
			Map<String, String> allIcs = HashMap.newHashMap(todoContainerUids.size());
			for (String todoContainerUid : todoContainerUids) {
				try (RestorableContainerItemBackupItem<User> restorable = repository
						.getRestorableUserTodo(restorableItem.domain(), restorableItem.entryUid(), todoContainerUid)) {
					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
					allIcs.put(cd.name, new String(restorable.dataStream().readAllBytes(), StandardCharsets.UTF_8));
				}
			}
			if (!allIcs.isEmpty()) {
				sendEmail.sendMessage(allIcs, bundle.getString("send.todo.restore.message"),
						bundle.getString("send.todo.restore.subject"));
			} else {
				logger.error("Nothing to send as email, no todolists found for {}@{}", restorableItem.entryUid(),
						restorableItem.domain());
			}
		} catch (Exception e) {
			logger.error("Error while export user todolists", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.TODOLISTS));
		} finally {
			restorableItem.endTask();
		}
	}
}
