/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.todolist;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.dataprotect.common.restore.CommonRestoreEntitiesFromDatabase;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.todolist.api.ITodoList;
import net.bluemind.todolist.api.VTodo;
import net.bluemind.todolist.api.VTodoChanges;

public class CommonRestoreTodos extends CommonRestoreEntitiesFromDatabase<VTodo> {

	public CommonRestoreTodos(IMonitoredRestoreRestorableItem item, BmContext back, BmContext live) {
		super(item, back, live);
	}

	public CommonRestoreTodos(IMonitoredRestoreRestorableItem item, BmContext live) {
		super(item, live);
	}

	@Override
	public void restoreEntities(List<String> allUids) {
		restoreEntities(allUids, item.entryUid(), item.liveEntryUid());
	}

	@Override
	public void restoreEntities(List<String> allUids, String backUid, String liveUid) {
		ITodoList backTodoApi = back.provider().instance(ITodoList.class, backUid);

		for (List<String> batch : Lists.partition(allUids, 1000)) {
			List<ItemValue<VTodo>> todos = backTodoApi.multipleGet(batch);
			restoreEntities(todos, liveUid);
		}
	}

	@Override
	public void restoreEntities(List<ItemValue<VTodo>> backVTodos, String liveUid) {
		ITodoList liveTodoApi = live.provider().instance(ITodoList.class, liveUid);

		VTodoChanges changes = VTodoChanges.create(backVTodos.stream()
				.map(e -> VTodoChanges.ItemAdd.create(e.uid, e.value, false)).collect(Collectors.toList()),
				Collections.emptyList(), Collections.emptyList());
		liveTodoApi.updates(changes);

		item.monitorLog("{} todos to restore ", backVTodos.size());
	}

}
