/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.todolist.impl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.common.backup.BackupContainerItemDescriptor;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.user.api.User;

public class TodolistBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/todolists");
	private static Logger logger = LoggerFactory.getLogger(TodolistBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	public TodolistBackupRepository(Path repository) {
		this.repositoryPath = new RepositoryBackupPath(repository);
	}

	record BackupTodoRecord(ContainerDescriptor cd, ByteBuf ics, ItemValue<User> user) {

	}

	private String readStream(ContainerDescriptor containerDescriptor, ByteBuf exportStream) throws IOException {
		Path icsFilePath = repositoryPath.getStoragePath(containerDescriptor, RepositoryBackupPath.ICS_FILE_EXTENSION);
		try (SeekableByteChannel out = Files.newByteChannel(icsFilePath, StandardOpenOption.CREATE,
				StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING)) {
			out.write(exportStream.nioBuffer());
		}
		return RepositoryBackupPath.getFilePathWithDirectParent(icsFilePath);
	}

	private String writeTodoListsContainerDescriptor(BackupTodoRecord todoRecord) throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(todoRecord.cd);
		JsonReadWrite.writeUser(jsonPathToWrite,
				new BackupContainerItemDescriptor<User>(todoRecord.cd, todoRecord.user));
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	public List<String> writeTodos(List<BackupTodoRecord> backupTodos) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupTodoRecord entryJson : backupTodos) {
			ret.add(writeTodoListsContainerDescriptor(entryJson));
			ret.add(readStream(entryJson.cd, entryJson.ics));
		}
		return ret;
	}

	/**
	 * Returns the list of todolists uids available in the backup
	 *
	 * @param ownerUid
	 * @return List of uid of todolists
	 * @throws IOException
	 */
	public List<String> getRestorableTodolistUids(String domainUid, String ownerUid) throws IOException {
		return repositoryPath.getRestorableUserContainerUids(domainUid, ownerUid);
	}

	private RestorableContainerItemBackupItem<User> extractUserTodolistDataFromZip(String folderName,
			String todoListUid, Path zipFilePath) {

		Map<String, ByteBuf> filesFromZip = new HashMap<>();
		try {
			filesFromZip = JsonReadWrite.findFilesInZipFile(zipFilePath, folderName, todoListUid,
					Arrays.asList(RepositoryBackupPath.JSON_FILE_EXTENSION, RepositoryBackupPath.ICS_FILE_EXTENSION));

		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		RestorableContainerItemBackupItem<User> todoContainer = null;
		ByteBuf jsonData = filesFromZip.get(RepositoryBackupPath.JSON_FILE_EXTENSION);
		if (jsonData != null) {
			todoContainer = new RestorableContainerItemBackupItem<>(JsonReadWrite.readerUser().read(jsonData));
			ByteBuf icsData = filesFromZip.get(RepositoryBackupPath.ICS_FILE_EXTENSION);
			if (icsData != null) {
				todoContainer.readDataStreamContent(icsData);
			}
		}
		return todoContainer;
	}

	private Optional<RestorableContainerItemBackupItem<User>> readZipFilesForUserTodo(String domainUid, String ownerUid,
			String todoListUid) throws Exception {
		return repositoryPath.getZipFilePathForOwner(domainUid, ownerUid).map(zipFilePath -> {
			return extractUserTodolistDataFromZip(ownerUid, todoListUid, zipFilePath);
		});
	}

	private RestorableContainerItemBackupItem<User> readJsonFilesForUserTodo(String ownerUid, String todoListUid)
			throws IOException {
		Path jsonPath = repositoryPath.resolveUserPathForContainer(ownerUid, todoListUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path icsPath = repositoryPath.resolveUserPathForContainer(ownerUid, todoListUid,
				RepositoryBackupPath.ICS_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new RestorableContainerItemBackupItem<>(JsonReadWrite.readerUser().read(in.readAllBytes()), icsPath);
		}
	}

	public RestorableContainerItemBackupItem<User> getRestorableUserTodo(String domainUid, String ownerUid,
			String todoListUid) throws Exception {
		Optional<RestorableContainerItemBackupItem<User>> restoredTodo = readZipFilesForUserTodo(domainUid, ownerUid,
				todoListUid);
		if (restoredTodo.isPresent()) {
			return restoredTodo.get();
		}

		return readJsonFilesForUserTodo(ownerUid, todoListUid);
	}
}
