/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.todolist.impl.pg;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DPError.DPKind;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.todolist.CommonRestoreTodos;
import net.bluemind.todolist.api.ITodoList;
import net.bluemind.todolist.api.ITodoUids;

public class RestoreTodolistsTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(RestoreTodolistsTaskPg.class);

	private final DataProtectGeneration backup;
	private final IMonitoredRestoreRestorableItem restorableItem;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	public RestoreTodolistsTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item) {
		this.backup = backup;
		this.restorableItem = item;
	}

	public void run() throws Exception {
		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();

			List<ContainerDescriptor> backLists = back.provider().instance(IContainers.class)
					.all(ContainerQuery.ownerAndType(restorableItem.entryUid(), ITodoUids.TYPE));

			restorableItem.monitorLog("Restoring '{}' todolist(s) from uid={}", backLists.size(),
					restorableItem.entryUid());
			logger.info("Backup contains {} todolist(s)", backLists.size());

			CommonRestoreTodos restoreTodos = new CommonRestoreTodos(restorableItem, back, live);
			for (ContainerDescriptor backList : backLists) {
				restorableItem.monitorSubWork(1);
				restore(restoreTodos, backList);
			}
		} catch (Exception e) {
			logger.error("Error while restoring todolists", e);
			restorableItem.errors().add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPKind.TODOLISTS));
		} finally {
			restorableItem.endTask();
		}
	}

	private void restore(CommonRestoreTodos restoreTodos, ContainerDescriptor backList) {

		ITodoList backupListApi = restoreTodos.back.provider().instance(ITodoList.class, backList.uid);
		List<String> allUids = backupListApi.allUids();
		restorableItem.monitorLog("Restoring {} [uid={}]", backList.name, backList.uid);

		String listUid = mapListUid(backList.uid);
		reset(restoreTodos, backList, listUid);
		restoreTodos.restoreEntities(allUids, backList.uid, listUid);
	}

	private void reset(CommonRestoreTodos restoreBooks, ContainerDescriptor backAB, String bookUid) {
		List<ContainerDescriptor> liveABs = restoreBooks.live.provider().instance(IContainers.class)
				.all(ContainerQuery.ownerAndType(restorableItem.liveEntryUid(), ITodoUids.TYPE));

		if (liveABs.stream().noneMatch(c -> c.uid.equals(bookUid))) {
			backAB.owner = restorableItem.liveEntryUid();
			restoreBooks.live.provider().instance(IContainers.class).create(bookUid, backAB);
			restorableItem.monitorLog("addressbook recreated");
		}
	}

	private String mapListUid(String uid) {
		if (!restorableItem.entryUid().equals(restorableItem.liveEntryUid())
				&& uid.endsWith(String.format("_%s", restorableItem.entryUid()))) {
			return String.format("%s%s", uid.substring(0, uid.length() - restorableItem.entryUid().length()),
					restorableItem.liveEntryUid());
		}

		return uid;
	}
}
