/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.todolist.impl;

import java.nio.charset.StandardCharsets;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.todolist.CommonRestoreTodos;
import net.bluemind.dataprotect.todolist.RestoreTodolistsWorker;
import net.bluemind.dataprotect.todolist.impl.pg.RestoreTodolistsTaskPg;
import net.bluemind.todolist.adapter.VTodoAdapter;
import net.bluemind.todolist.api.VTodo;
import net.bluemind.user.api.User;

public class RestoreTodoListsTask extends AbstractRestoreTodoLists {
	private static final Logger logger = LoggerFactory.getLogger(RestoreTodoListsTask.class);

	RestoreTodolistsWorker worker;

	public RestoreTodoListsTask(DataProtectGeneration backup, Restorable item) {
		super(backup, item);
		worker = new RestoreTodolistsWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.TODOLISTS.name()));

		TodolistBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreTodolistsTaskPg(backup, restorableItem).run();
		}
	}

	private void restore(TodolistBackupRepository repository) {
		try {
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
			List<String> todolistUids = repository.getRestorableTodolistUids(restorableItem.domain(),
					restorableItem.entryUid());
			for (String todoListUid : todolistUids) {
				try (RestorableContainerItemBackupItem<User> restorable = repository
						.getRestorableUserTodo(restorableItem.domain(), restorableItem.entryUid(), todoListUid)) {
					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();

					List<ItemValue<VTodo>> todosBack = new VTodoAdapter().convertToVTodoList(
							new String(restorable.dataStream().readAllBytes(), StandardCharsets.UTF_8));
					restorableItem.monitorLog("Restore todolist {}@{} ({})", cd.name, cd.domainUid, todoListUid);

					worker.createOrUpdateLiveEntry(ItemValue.create(todoListUid, cd));

					new CommonRestoreTodos(restorableItem, live).restoreEntities(todosBack, todoListUid);
				}
			}
		} catch (Exception e) {
			logger.error("Error while restoring todolists", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.TODOLISTS));
		} finally {
			restorableItem.endTask();
		}
	}

}
