/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2025
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.shared.mailbox.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DPError.DPKind;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.directory.DirectoryBackupRepository;
import net.bluemind.dataprotect.shared.mailbox.impl.pg.RestoreSharedMailboxTaskPg;
import net.bluemind.tag.api.TagChanges;

public class RestoreSharedMailboxTask extends BlockingServerTask implements IServerTask {

	private static final Logger logger = LoggerFactory.getLogger(RestoreSharedMailboxTask.class);

	protected final DataProtectGeneration backup;
	protected IMonitoredRestoreRestorableItem restorableItem;
	protected final SharedMailboxRestoreWorker sharedMailboxWorker;

	public RestoreSharedMailboxTask(DataProtectGeneration backup, Restorable item) {
		this.backup = backup;
		this.restorableItem = new RestoreRestorableItem(item);
		this.sharedMailboxWorker = new SharedMailboxRestoreWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPKind.SHARED_MAILBOX.name()));

		DirectoryBackupRepository repository = DirectoryBackupRepository.getRepository(backup);
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreSharedMailboxTaskPg(backup, restorableItem, sharedMailboxWorker).run();
		}
	}

	private void restore(DirectoryBackupRepository repository) {
		try {
			repository.getRestorableUser(restorableItem.domain(), restorableItem.entryUid()).ifPresentOrElse(rn -> {
				try {
					sharedMailboxWorker.createOrUpdateLiveEntry(rn.item);
					sharedMailboxWorker.restoreSharedMailboxSettings(rn.settings);
					sharedMailboxWorker.restoreSharedMailboxTags(
							rn.tags.stream().map(t -> TagChanges.ItemAdd.create(t.uid, t.value)).toList());
					sharedMailboxWorker.restoreSharedMailboxFilters(rn.filter);
					sharedMailboxWorker.restoreGroupMembership(rn.memberOf);
					sharedMailboxWorker.restoreEntryDependantContainers();
				} catch (InterruptedException e) {
					logger.error(
							"Error while restoring shared mailbox, when trying to delete first existing shared mailbox",
							e);
					restorableItem.errors().add(
							DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.SHARED_MAILBOX));
				}
			}, () -> logger.warn("Error while restoring shared mailbox {}, because not found",
					restorableItem.entryUid()));

		} catch (Exception e) {
			logger.error("Error while restoring shared mailbox", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.SHARED_MAILBOX));
		} finally {
			restorableItem.endTask();
		}
	}

}
