/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2025
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.shared.mailbox.impl.pg;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DPError.DPKind;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.shared.mailbox.impl.SharedMailboxRestoreWorker;
import net.bluemind.group.api.Group;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.TagChanges;
import net.bluemind.tag.api.TagChanges.ItemAdd;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.User;

public class RestoreSharedMailboxTaskPg {

	private static final Logger logger = LoggerFactory.getLogger(RestoreSharedMailboxTaskPg.class);
	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;
	private final SharedMailboxRestoreWorker sharedMailboxWorker;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	public RestoreSharedMailboxTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item,
			SharedMailboxRestoreWorker sharedMailboxWorker) {
		this.backup = backup;
		this.restorableItem = item;
		this.sharedMailboxWorker = sharedMailboxWorker;
	}

	public void run() {
		logger.info("Restoring shared mailbox {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext backupContext = bdp.createContextWithData(backup, restorableItem.item());
			IServiceProvider back = backupContext.provider();

			IUser sharedMailboxService = back.instance(IUser.class, restorableItem.domain());
			ItemValue<User> backupUser = sharedMailboxService.getComplete(restorableItem.entryUid());
			List<ItemValue<Group>> memberOf = sharedMailboxService.memberOf(restorableItem.entryUid());

			ItemValue<User> liveUser = sharedMailboxWorker.createOrUpdateLiveEntry(backupUser);
			restorableItem.setLiveEntryUid(liveUser.uid);

			IUserSettings settings = back.instance(IUserSettings.class, restorableItem.domain());
			sharedMailboxWorker.restoreSharedMailboxSettings(settings.get(restorableItem.entryUid()));

			ITags tagsBackup = back.instance(ITags.class, ITagUids.TYPE + "_" + restorableItem.entryUid());
			List<TagChanges.ItemAdd> sharedMailboxTags = tagsBackup.all().stream()
					.map(tag -> ItemAdd.create(tag.uid, tag.value)).toList();
			sharedMailboxWorker.restoreSharedMailboxTags(sharedMailboxTags);

			IMailboxes mboxesBackup = back.instance(IMailboxes.class, restorableItem.domain());
			sharedMailboxWorker.restoreSharedMailboxFilters(mboxesBackup.getMailboxFilter(restorableItem.entryUid()));
			sharedMailboxWorker.restoreGroupMembership(memberOf);
			sharedMailboxWorker.restoreEntryDependantContainers();

			restorableItem.monitorLog(
					"shared mailbox " + restorableItem.entryUid() + ":" + restorableItem.displayName() + " restored",
					"");
		} catch (Exception e) {
			logger.error("Error while restoring shared mailbox", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPKind.SHARED_MAILBOX));
		} finally {
			restorableItem.endTask();
		}

	}

}
