/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.service.tool;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public final class CoreCommonBackupWorker {
	private static Logger logger = LoggerFactory.getLogger(CoreCommonBackupWorker.class);

	public static boolean supportsTag(String tag) {
		return TagDescriptor.bm_core.getTag().equals(tag);
	}

	public static ByteBuf mmapOutput() {
		try {
			Path backingFile = Files.createTempFile("dp-worker", ".output");
			int capacity = Integer.MAX_VALUE;
			try (RandomAccessFile raf = new RandomAccessFile(backingFile.toFile(), "rw")) {
				raf.setLength(Integer.MAX_VALUE);
				MappedByteBuffer targetBuffer = raf.getChannel().map(MapMode.READ_WRITE, 0, capacity);
				ByteBuf wrapped = Unpooled.wrappedBuffer(targetBuffer);
				wrapped.writerIndex(0).readerIndex(0);
				Files.delete(backingFile);
				return wrapped;
			}
		} catch (IOException e) {
			throw new ServerFault(e);
		}
	}

	public static void prepareIndexJson(Path tmpFolder, Path outputFolder, Map<String, ByteBuf> jsonFiles)
			throws IOException {
		Path tmpIndexJsonPath = tmpFolder.resolve("index.json");
		try (InputStream instream = Files.newInputStream(tmpIndexJsonPath)) {
			ByteBuf jsonBuff = CoreCommonBackupWorker.mmapOutput();
			jsonBuff.writeBytes(instream.readAllBytes());
			jsonFiles.put(outputFolder.resolve("index.json").toString(), jsonBuff);
		}
	}

	public static void writeJsonFiles(IDPContext ctx, Map<String, ByteBuf> jsonFiles) {
		jsonFiles.entrySet().forEach(jsonFile -> {
			ByteBuf zipBuf = CoreCommonBackupWorker.mmapOutput();
			zipBuf.writeBytes(jsonFile.getValue());
			try {
				try (ByteBufInputStream in = new ByteBufInputStream(zipBuf);
						OutputStream outStream = new FileOutputStream(new File(jsonFile.getKey()))) {
					outStream.write(in.readAllBytes());
				}
			} catch (IOException ioe) {
				ctx.error(ioe, "Unable to write json file {}", jsonFile.getKey());
			}
		});
	}

	public static void writeZipFiles(IDPContext ctx, Path outputPath, List<ZipBuilder> zipBuilderList) {
		zipBuilderList.forEach(zipBuilder -> {
			Path zipFileOutputPath = outputPath.resolve(zipBuilder.zipFileName);
			try {
				zipBuilder.writeToFile(zipFileOutputPath);
			} catch (IOException ioe) {
				ctx.error(ioe, "Unable to write zip file {}", zipFileOutputPath);
			}
		});

	}

	public static MailboxIndexJsonList readIndexJson(Path workingPath) throws IOException {
		Path indexPath = workingPath.resolve("index.json");
		if (!Files.exists(indexPath)) {
			throw new FileNotFoundException("index.json file not found: " + indexPath);
		}

		try (InputStream in = Files.newInputStream(indexPath)) {
			return JsonUtils.read(in, MailboxIndexJsonList.class);
		} catch (IOException e) {
			logger.error("Unable to read index.json file", e);
		}

		return null;
	}

	public static boolean zipFileFilter(Path p) {
		return p.getFileName().toString().endsWith(".zip");
	}

	public static List<ZipBuilder> createZipBufferList(List<String> zipFiles) {
		return zipFiles.stream().map(ZipBuilder::new).toList();
	}

	public static void prepareDataDirs(IDPContext ctx, ItemValue<Server> toBackup, Path tmpFolder, Path outputFolder)
			throws IOException {
		// The node is required because we want files to be owned by root
		// even in JUnit tests
		INodeClient nc = NodeActivator.get(toBackup.value.address());
		nc.mkdirs(outputFolder.toString());

		ctx.info(String.format("Start copiing backup files from '%s' to '%s'", tmpFolder, outputFolder));

		writeUserFilesToOutputPath(ctx, tmpFolder, outputFolder);
		writeDomainFilesToOutputPath(ctx, tmpFolder, outputFolder);
	}

	private static void writeDomainFilesToOutputPath(IDPContext ctx, Path tmpFolder, Path outputFolder)
			throws IOException {
		Path tmpDomainFolder = tmpFolder.resolve("domain");
		if (!Files.exists(tmpDomainFolder)) {
			return;
		}

		Path targetDomainFolder = outputFolder.resolve("domain");
		if (!Files.exists(targetDomainFolder)) {
			Files.createDirectories(targetDomainFolder);
		}

		ctx.info(String.format("Start write domain json files from '%s' to '%s'", tmpFolder, outputFolder));

		try (Stream<Path> stream = Files.walk(tmpDomainFolder)) {
			stream.forEach(p -> {
				Path relativePath = tmpDomainFolder.relativize(p);
				Path targetPath = targetDomainFolder.resolve(relativePath);
				try {
					if (Files.isDirectory(p)) {
						Files.createDirectories(targetPath);
					} else {
						try (InputStream instream = Files.newInputStream(p);
								OutputStream outStream = new FileOutputStream(new File(targetPath.toString()))) {
							outStream.write(instream.readAllBytes());
						}
					}
				} catch (ServerFault | IOException e) {
					logger.error("Unable to copy {} to {}: {}", p, targetPath, e.getMessage());
					ctx.error(e, "Unable to copy {} to {}", p, targetPath);
				}
			});
		}
	}

	private static void writeUserFilesToOutputPath(IDPContext ctx, Path tmpFolder, Path outputFolder)
			throws IOException {
		Path tmpUserFolder = tmpFolder.resolve("user");
		Path targetUserFolder = outputFolder.resolve("user");
		if (!Files.exists(targetUserFolder)) {
			Files.createDirectories(targetUserFolder);
		}

		ctx.info(String.format("Start write json files to zip files from '%s' to '%s'", tmpFolder, outputFolder));

		MailboxIndexJsonList indexJson = CoreCommonBackupWorker.readIndexJson(tmpUserFolder);
		if (indexJson == null) {
			ctx.error("Unable to read index.json from '{}' for backup", tmpUserFolder.toString());
			return;
		}

		Map<String, ByteBuf> jsonFiles = new HashMap<>();
		List<ZipBuilder> zipBufferList = CoreCommonBackupWorker.createZipBufferList(
				indexJson.mailboxesIndexInfo().stream().map(i -> i.zipFileName).distinct().toList());

		try {
			indexJson.mailboxesIndexInfo().forEach(mailbox -> {
				try {
					writeMailboxIndexJson(ctx, mailbox, zipBufferList, tmpUserFolder);
				} catch (FileNotFoundException e) {
					ctx.error(e, e.getMessage());
				}
			});

			prepareIndexJson(tmpUserFolder, targetUserFolder, jsonFiles);

		} finally {
			CoreCommonBackupWorker.writeJsonFiles(ctx, jsonFiles);
			CoreCommonBackupWorker.writeZipFiles(ctx, targetUserFolder, zipBufferList);
			ctx.info(String.format("backup part done in '%s'", outputFolder));
		}
	}

	public static void writeMailboxIndexJson(IDPContext ctx, MailboxIndexJson mailbox, List<ZipBuilder> zipBufferList,
			Path tmpFolder) throws FileNotFoundException {
		zipBufferList.stream().filter(zipBuild -> zipBuild.zipFileName.equals(mailbox.zipFileName)).findFirst()
				.ifPresentOrElse(zipBuilder -> {
					AtomicBoolean folderCreated = new AtomicBoolean();
					mailbox.filenames.forEach(mailboxFile -> {
						Path jsonFilePath = tmpFolder.resolve(mailboxFile);
						Path path = Paths.get(mailboxFile);
						try {
							if (path.getParent() != null && !folderCreated.get()) {
								zipBuilder.addDirectoryEntry(path.getParent().getFileName().toString());
								folderCreated.set(true);
							}
							zipBuilder.addFileEntry(mailboxFile, jsonFilePath);
						} catch (IOException e) {
							ctx.error(e, "Unable to write {} to zip {} with entry name {}", jsonFilePath,
									mailbox.zipFileName, path.getFileName().toString());
						}
					});

				}, () -> new FileNotFoundException(mailbox.zipFileName + " not found, cannot write into"));
	}

}
