/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.service.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonValue;

import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.api.GenerationIndex;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.service.internal.IDirEntriesWorker.DirEntries;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.utils.FileUtils;

public class CommonBackupWorker implements ICommonBackupWorker {
	private static Logger logger = LoggerFactory.getLogger(CommonBackupWorker.class);
	private static final int PARTITIONS = 32;

	protected IDPContext ctx;
	protected PartGeneration partGen;

	public CommonBackupWorker(IDPContext ctx, PartGeneration partGen) {
		this.ctx = ctx;
		this.partGen = partGen;
	}

	protected IServiceProvider provider() {
		return SPResolver.get().resolve(null);
	}

	@Override
	public void logError(Throwable throwable) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(throwable, throwable.getMessage());
	}

	@Override
	public void logError(Exception ex, String message, Object... args) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(ex, message, args);
	}

	@Override
	public void logError(String message, Object... args) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error("en", message, args);
	}

	public record MailboxIndexJsonList(@JsonValue List<MailboxIndexJson> mailboxesIndexInfo) {
	}

	public static String zipFilename(String prefix, String zipHash) {
		return prefix + "-" + zipHash + ".zip";
	}

	public MailboxIndexJsonList runIt(String domainUid, List<String> dirEntriesUids) {
		if (dirEntriesUids.isEmpty()) {
			return new MailboxIndexJsonList(Collections.emptyList());
		}

		long timeout = CoreConfig.get().getLong(CoreConfig.DataProtect.BACKUP_INDIVIDUAL_PART_TIME);
		List<MailboxIndexJson> zipIndexOwnersMap = new ArrayList<>();
		Map<Integer, List<String>> partitionedDirEntriesList = dirEntriesUids.stream()
				.collect(Collectors.groupingBy(uid -> Math.abs(uid.hashCode() % PARTITIONS)));

		try (var executor = Executors.newFixedThreadPool(
				Math.min(partitionedDirEntriesList.size(), Runtime.getRuntime().availableProcessors()))) {

			var futures = partitionedDirEntriesList.entrySet().stream()
					.map(partition -> CompletableFuture
							.runAsync(() -> zipIndexOwnersMap.addAll(partTreatment(domainUid, partition)), executor))
					.toList();

			CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).orTimeout(timeout, TimeUnit.HOURS)
					.join();
		}

		return new MailboxIndexJsonList(zipIndexOwnersMap);
	}

	private List<MailboxIndexJson> partTreatment(String domainUid, Entry<Integer, List<String>> partition) {
		String zipHash = String.valueOf(partition.getKey());
		return partitionTreatment(domainUid, zipHash, partition.getValue());
	}

	public static List<GenerationIndex> loadDirEntries(String domainUid) {
		String path = "/var/backups/bluemind/directory";
		List<GenerationIndex> dirEntries = new ArrayList<>();

		File dir = new File(path);
		if (!dir.exists()) {
			throw new ServerFault("Unable to load directory listing: '" + path + "' does not exists");
		}

		Stream.of(dir.listFiles(File::isFile)).filter(f -> f.getName().equalsIgnoreCase("___" + domainUid + ".json"))
				.findFirst().ifPresentOrElse(f -> {
					logger.debug("Read domain file {}", f);
					logger.debug("Consuming for domain {}", domainUid);

					try (InputStream in = new FileInputStream(f)) {
						String data = FileUtils.streamString(in, true);
						dirEntries.addAll(JsonUtils.read(data, DirEntries.class).entries());
					} catch (Exception e) {
						throw new ServerFault("Cannot read domain json file " + f.getName(), e);
					}
				}, () -> new ServerFault("Unable to read directory file: '" + path + "'"));

		return dirEntries;
	}

}
