/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.sdsspool;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonObject;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.rest.base.JsonStreams;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.sdsspool.SdsSpoolDownloader.RefusedTransfer;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.sds.sync.api.ISdsSync;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.helper.ArchiveHelper;
import net.bluemind.system.sysconf.helper.SysConfHelper;
import net.bluemind.utils.ProgressPrinter;

public class SdsSpoolWorker implements IBackupWorker {
	private final Path rootPath;
	private final SdsDataProtectSpool dataprotectSpool;
	private static final Logger logger = LoggerFactory.getLogger(SdsSpoolWorker.class);
	AtomicLong lastSavedOffset = new AtomicLong(0);
	AtomicLong lastSavedOffsetNanos = new AtomicLong(0);
	AtomicLong sdsSpoolTransferredBytes = new AtomicLong(0);

	public SdsSpoolWorker() {
		rootPath = SdsDataProtectSpool.DEFAULT_PATH.getParent();
		dataprotectSpool = new SdsDataProtectSpool(SdsDataProtectSpool.DEFAULT_PATH);
	}

	@Override
	public long getTransferredMegaBytes() {
		return (long) Math.round(sdsSpoolTransferredBytes.get() / (1024.0 * 1024.0));
	}

	@Override
	public String getDataType() {
		return WorkerDataType.SDS_SPOOL.value;
	}

	private void initSpoolDirectories(INodeClient nc) {
		List<String> suffix = new LinkedList<>();
		for (char c = 'a'; c <= 'f'; c++) {
			suffix.add("" + c);
		}
		for (char c = '0'; c <= '9'; c++) {
			suffix.add("" + c);
		}
		int len = suffix.size();
		List<String> realSuffix = new ArrayList<>(len * len * len);
		for (int i = 0; i < suffix.size(); i++) {
			for (int j = 0; j < suffix.size(); j++) {
				realSuffix.add(suffix.get(i) + "/" + suffix.get(j));
			}
		}

		for (String s : realSuffix) {
			nc.mkdirs("/var/backups/bluemind/sds-spool/spool/" + s);
		}
	}

	@Override
	public void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup) throws ServerFault {
		SystemConf sysconf = SysConfHelper.fromSharedMap();
		if (!ArchiveHelper.isSdsArchiveKind(sysconf)) {
			return;
		}

		if (sysconf.stringList(SysConfKeys.dataprotect_skip_datatypes.name())
				.contains(WorkerDataType.SDS_SPOOL.value)) {
			return;
		}

		long startDate = Instant.now().getEpochSecond();
		ctx.info("Starting message bodies backup");

		INodeClient nc = NodeActivator.get(toBackup.value.address());
		nc.mkdirs(dataprotectSpool.path().toString());
		initSpoolDirectories(nc);

		ISdsSync sdsSyncApi = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(ISdsSync.class);

		AtomicLong lastIndex = new AtomicLong(readLastIndex(ctx, nc));
		lastSavedOffset.set(lastIndex.get());
		sdsSpoolTransferredBytes.set(0);
		Stream sdsSyncStream = sdsSyncApi.sync(lastIndex.get());
		ProgressPrinter progress = ProgressPrinter.createWithPercent(sdsSyncApi.count(lastIndex.get()), 10);
		ctx.info(String.format("Start download message bodies %s", progress.toString()));
		AtomicBoolean shouldAbort = new AtomicBoolean(false);
		Config coreConfig = CoreConfig.get();

		SdsSpoolDownloader downloader = new SdsSpoolDownloader((successOffset, minimumBatchOffset, nothingInFlight) -> {
			long now = System.nanoTime();
			if (nothingInFlight || (now - lastSavedOffsetNanos.get() > Duration.ofSeconds(1).toNanos())) {
				lastSavedOffsetNanos.set(now);
				saveLastIndex(nc, Math.max(minimumBatchOffset, successOffset));
			}
		}, errorMessage -> {
			ctx.error("Unrecoverable error while downloading: {}", errorMessage);
			shouldAbort.set(true);
		}, coreConfig.getInt(CoreConfig.DataProtect.SDS_DOWNLOAD_CONCURRENCY));

		CompletableFuture<Void> future = consumeSdsSyncEvents(ctx, sdsSyncStream, progress, shouldAbort, downloader);

		CompletableFuture.allOf(future, downloader.asPromise()).exceptionally(throwable -> {
			shouldAbort.set(true);

			if (throwable instanceof TimeoutException timeout) {
				ctx.error(timeout, "Operation timed out after 22 hours of processing. Cancelling all tasks.");
				shouldAbort.set(true);
			} else {
				ctx.error(throwable, "Unknown error while downloading sds queue");
			}
			throw new CompletionException(throwable);
		}).join();

		sdsSpoolTransferredBytes.set(downloader.getDownloadedSize());
		ctx.info(String.format("Ending message bodies backup (%d MB) in %d seconds", getTransferredMegaBytes(),
				Instant.now().getEpochSecond() - startDate));
	}

	private CompletableFuture<Void> consumeSdsSyncEvents(IDPContext ctx, Stream sdsSyncStream, ProgressPrinter progress,
			AtomicBoolean shouldAbort, SdsSpoolDownloader downloader) {
		return JsonStreams.consume(VertxStream.read(sdsSyncStream), body -> {
			if (shouldAbort.get()) {
				throw new CancellationException("Operation was cancelled");
			}
			progress.add();
			long index = body.getLong("index");
			String type = body.getString("type");

			if (type.equals("BODYADD") || type.equals("BODYDEL")) {
				String guid = body.getString("key");
				String serverUid = body.getString("srv");

				if (type.equals("BODYADD")) {
					msgBodyAddTreatment(ctx, shouldAbort, downloader, index, guid, serverUid);
				} else if (type.equals("BODYDEL")) {
					msgBodyDeleteTreatment(guid);
				}

				if (progress.shouldPrint()) {
					ctx.info(String.format("Progress (total is not accurate): %s", progress.toString()));
				}
			}
		});
	}

	private void msgBodyDeleteTreatment(String guid) {
		Path toDel = dataprotectSpool.livePath(guid);
		try {
			Files.deleteIfExists(toDel);
		} catch (IOException e) {
			// ok
		}
	}

	private void msgBodyAddTreatment(IDPContext ctx, AtomicBoolean shouldAbort, SdsSpoolDownloader downloader,
			long index, String guid, String serverUid) {
		Path fp = dataprotectSpool.livePath(guid);
		try {
			downloader.addTransfer(serverUid, guid, fp, index);
		} catch (RefusedTransfer e) {
			ctx.error(e, "Refused transfer or interrupted");
			shouldAbort.set(true);
		}
	}

	private void saveLastIndex(INodeClient nc, long lastIndex) {
		logger.info("Saving last index: {}", lastIndex);
		JsonObject jo = new JsonObject();
		jo.put("index", lastIndex);
		nc.writeFile(rootPath.resolve("stream-index.json").toString(),
				new ByteArrayInputStream(jo.toBuffer().getBytes()));
	}

	private long readLastIndex(IDPContext ctx, INodeClient nc) {
		Path index = rootPath.resolve("stream-index.json");
		if (Files.notExists(index)) {
			return 0L;
		}
		String indexpath = index.toString();
		try {
			JsonObject jo = new JsonObject(Buffer.buffer(nc.read(indexpath)));
			return jo.getLong("index");
		} catch (Exception e) {
			ctx.error(e, "Unable to read last index from {}", indexpath);
			return 0L;
		}
	}

	@Override
	public Set<String> getDataDirs() {
		return Set.of(rootPath.toString());
	}

	@Override
	public boolean supportsTag(String tag) {
		return TagDescriptor.bm_core.getTag().equals(tag);
	}
}
