/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.dataprotect.resource.impl;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.calendar.impl.AbstractRestoreCalendar;
import net.bluemind.dataprotect.calendar.impl.CalendarBackupRepository;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.email.SendIcs;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.resource.impl.pg.SendResourceICSTaskPg;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;

public class SendResourceICSTask extends AbstractRestoreCalendar implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(SendResourceICSTask.class);

	private final RestoreActionExecutor<EmailData> executor;
	private final ResourceBundle bundle;

	@SuppressWarnings("unchecked")
	public SendResourceICSTask(DataProtectGeneration backup, Restorable item,
			RestoreActionExecutor<? extends IRestoreActionData> executor) {
		super(backup, item);
		this.executor = (RestoreActionExecutor<EmailData>) executor;
		this.bundle = ResourceBundle.getBundle("OSGI-INF/l10n/RestoreResource", Locale.of(ServerSideServiceProvider
				.getProvider(SecurityContext.SYSTEM).getContext().getSecurityContext().getLang()));
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startExportMsg(DPError.DPKind.RESOURCE.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting export for uid {}", restorableItem.entryUid());
		}

		CalendarBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new SendResourceICSTaskPg(backup, restorableItem, executor, bundle).run();
		}

	}

	protected void restore(CalendarBackupRepository repository) {
		SendIcs sendEmail = new SendIcs(restorableItem, executor);

		try {
			List<String> calendarUids = repository.getRestorableCalendarUids(restorableItem);
			Map<String, String> allIcs = HashMap.newHashMap(calendarUids.size());
			for (String calendarUid : calendarUids) {
				try (RestorableContainerItemBackupItem restorable = repository.getRestorableDirCalendar(restorableItem,
						calendarUid)) {
					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
					allIcs.put(cd.name, new String(restorable.dataStream().readAllBytes(), StandardCharsets.UTF_8));
				}
			}
			if (!allIcs.isEmpty()) {
				sendEmail.sendMessage(allIcs, bundle.getString("send.resource.restore.message"),
						bundle.getString("send.resource.restore.subject"));
			} else {
				logger.error("Nothing to send as email, no calendars found for {}@{}", restorableItem.entryUid(),
						restorableItem.domain());
			}
		} catch (Exception e) {
			logger.error("Error while export resource calendars", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.RESOURCE));
		} finally {
			restorableItem.endTask();
		}
	}
}
