/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2021
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.deserializer;

import java.io.IOException;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

public class CyrusSdsBackupMessage {
	protected static Logger logger = LoggerFactory.getLogger(CyrusSdsBackupMessage.class);

	public String guid;
	public Date date;
	public long itemId;
	public Set<String> flags;

	public CyrusSdsBackupMessage(String guid, Date date, long itemId, Set<String> flags) {
		this.guid = guid;
		this.date = date;
		this.itemId = itemId;
		this.flags = flags;
	}

	public static CyrusSdsBackupMessage from(DateTimeFormatter datefmt, JsonParser parser) throws IOException {
		Date date = null;
		String guid = null;
		long itemId = -1;
		Set<String> tags = new HashSet<>();

		if (parser.currentToken() != JsonToken.START_OBJECT) {
			throw new IllegalStateException("Expected an object");
		}
		JsonToken tok;
		while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
			String fieldName = parser.currentName();
			switch (fieldName) {
			case "g":
				guid = parser.nextTextValue();
				break;
			case "d":
				date = new Date(Instant.from(datefmt.parse(parser.nextTextValue())).toEpochMilli());
				break;
			case "i":
				itemId = parser.nextLongValue(0);
				break;
			case "f":
				parser.nextToken();
				if (parser.currentToken() != JsonToken.START_ARRAY) {
					throw new IllegalStateException("Expected an array");
				}
				while (parser.nextToken() != JsonToken.END_ARRAY) {
					String tag = parser.getText();
					if (tag != null) {
						tags.add(tag);
					} else {
						break;
					}
				}

			default:
				break;
			}
		}
		if (guid != null && date != null) {
			return new CyrusSdsBackupMessage(guid, date, itemId, tags);
		} else {
			return null;
		}
	}

	@Override
	public String toString() {
		return guid;
	}
}