/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2021
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.deserializer;

import java.io.IOException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

public class CyrusSdsBackupFolder {
	public static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")
			.withZone(ZoneId.of("GMT"));

	private String uid;
	private String name;
	private String fullName;
	private long changeSetVersion;
	private Date created;
	private List<CyrusSdsBackupMessage> messages = new ArrayList<>();

	public CyrusSdsBackupFolder(String uid, String name, String fullName, long changeSetVersion, Date created) {
		this.uid = uid;
		this.fullName = fullName;
		this.name = name;
		this.changeSetVersion = changeSetVersion;
		this.created = created;
	}

	public List<CyrusSdsBackupMessage> messages() {
		return messages;
	}

	public int messageCount() {
		return messages.size();
	}

	public String toString() {
		return String.format("<CyrusSdsBackupFolder uid=%s name=%s fullName=%s>", uid, name, fullName);
	}

	public String uid() {
		return uid;
	}

	public String name() {
		return name;
	}

	public long version() {
		return changeSetVersion;
	}

	public Date created() {
		return created;
	}

	public String fullName() {
		return fullName;
	}

	public String fullNameWithoutInbox() {
		return fullNameWithoutInbox("INBOX");
	}

	public String fullNameWithoutInbox(String inboxname) {
		return fullName.replace(inboxname, "");
	}

	public static CyrusSdsBackupFolder from(JsonParser parser) throws IOException {
		if (parser.currentToken() != JsonToken.START_OBJECT) {
			throw new IllegalStateException("Expected an object");
		}
		String fullName = null;
		String uid = null;
		String name = null;
		Date created = null;
		long changeSetVersion = -1;
		CyrusSdsBackupFolder folder = null;

		JsonToken tok;
		while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
			String fieldName = parser.currentName();
			switch (fieldName) {
			case "fullName":
				fullName = parser.nextTextValue();
				if (uid != null && fullName != null && name != null) {
					folder = new CyrusSdsBackupFolder(uid, name, fullName, changeSetVersion, created);
				}
				break;
			case "name":
				name = parser.nextTextValue();
				if (uid != null && fullName != null && name != null) {
					folder = new CyrusSdsBackupFolder(uid, name, fullName, changeSetVersion, created);
				}
				break;
			case "uid":
				uid = parser.nextTextValue();
				if (uid != null && fullName != null && name != null) {
					folder = new CyrusSdsBackupFolder(uid, name, fullName, changeSetVersion, created);
				}
				break;
			case "changeSetVersion":
				changeSetVersion = parser.nextLongValue(0);
				if (uid != null && fullName != null && name != null) {
					folder = new CyrusSdsBackupFolder(uid, name, fullName, changeSetVersion, created);
				}
				break;
			case "created":
				created = new Date(Instant.from(DATE_FORMAT.parse(parser.nextTextValue())).toEpochMilli());
				if (uid != null && fullName != null && name != null) {
					folder = new CyrusSdsBackupFolder(uid, name, fullName, changeSetVersion, created);
				}
				break;
			case "messages":
				if (folder == null) {
					throw new IllegalStateException("expected folder to be fully defined");
				}
				parser.nextToken();
				if (parser.currentToken() != JsonToken.START_ARRAY) {
					throw new IllegalStateException("Expected an array");
				}
				while ((tok = parser.nextToken()) != JsonToken.END_ARRAY && tok != null) {
					CyrusSdsBackupMessage msg = CyrusSdsBackupMessage.from(DATE_FORMAT, parser);
					if (msg != null) {
						folder.addMessage(msg);
					}
				}
				break;
			default:
				break;
			}

		}
		return folder;
	}

	public void addMessage(CyrusSdsBackupMessage msg) {
		messages.add(msg);
	}
}