/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.dataprotect.mailbox.mailshare;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.directory.DirectoryBackupRepository;
import net.bluemind.dataprotect.mailbox.MboxRestoreService.Mode;

public class RestoreMailshareTask extends BlockingServerTask implements IServerTask {

	private static final Logger logger = LoggerFactory.getLogger(RestoreMailshareTask.class);
	private final DataProtectGeneration backup;
	private final MailshareRestoreWorker mailshareWorker;
	private IMonitoredRestoreRestorableItem restorableItem;

	public RestoreMailshareTask(DataProtectGeneration dpg, Restorable box, Mode m) {
		if (dpg == null) {
			throw new NullPointerException("DataProtectGeneration can't be null");
		}
		this.backup = dpg;
		this.restorableItem = new RestoreRestorableItem(box);
		this.mailshareWorker = new MailshareRestoreWorker(dpg, restorableItem, m);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.MAILSHARE.name()));
		DirectoryBackupRepository repository = DirectoryBackupRepository.getRepository(backup);
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreMailshareTaskPg(backup, restorableItem, mailshareWorker).run();
		}

	}

	protected void restore(DirectoryBackupRepository repository) {
		try {
			logger.info("Start restoring mailshare {}", restorableItem.entryUid());
			repository.getRestorableMailshare(restorableItem.domain(), restorableItem.entryUid()).ifPresent(rn -> {
				try {
					mailshareWorker.createOrUpdateLiveEntry(rn.item);
					mailshareWorker.restoreMailbox();
				} catch (InterruptedException e) {
					logger.error("Error while trying to delete first existing mailshare", e);
					restorableItem.errors()
							.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.MAILSHARE));
				}
			});

		} catch (Exception e) {
			logger.error("Error while restoring mailshare {}:{}@{}: {}", restorableItem.entryUid(),
					restorableItem.displayName(), restorableItem.domain(), e.getMessage());
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.MAILSHARE));
		} finally {
			restorableItem.endTask();
		}
	}

}
