/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.mailshare;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.core.task.service.TaskUtils;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.directory.RestoreDirWorker;
import net.bluemind.dataprotect.mailbox.MboxRestoreService.Mode;
import net.bluemind.dataprotect.mailbox.RestoreBoxTask;
import net.bluemind.mailshare.api.IMailshare;
import net.bluemind.mailshare.api.Mailshare;

public class MailshareRestoreWorker extends RestoreDirWorker<Mailshare> {

	private final Mode mode;

	public MailshareRestoreWorker(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem, Mode mode) {
		super(backup, restorableItem);
		this.mode = mode;
	}

	@Override
	public ItemValue<Mailshare> createOrUpdateLiveEntry(ItemValue<Mailshare> restorableMailshare)
			throws InterruptedException {

		IMailshare mailshareServiceLive = live.instance(IMailshare.class, restorableItem.domain());

		ItemValue<Mailshare> liveMailshare = mailshareServiceLive.getComplete(restorableItem.entryUid());
		if (liveMailshare != null) {
			mailshareServiceLive.update(liveMailshare.uid, restorableMailshare.value);
			restorableItem.setLiveEntryUid(liveMailshare.uid);
			return liveMailshare;
		}

		liveMailshare = mailshareServiceLive.byEmail(restorableMailshare.value.defaultEmailAddress());
		if (liveMailshare != null) {
			TaskRef deleteTask = mailshareServiceLive.delete(liveMailshare.uid);
			TaskStatus taskDeleteStatus = TaskUtils.waitForInterruptible(live, deleteTask);
			if (taskDeleteStatus.state.succeed) {
				restorableItem.monitorLog("Delete mailshare '{}' succeed {}",
						restorableMailshare.value.defaultEmailAddress(), taskDeleteStatus.result);
			}
		}

		mailshareServiceLive.create(restorableItem.entryUid(), restorableMailshare.value);
		liveMailshare = mailshareServiceLive.getComplete(restorableItem.entryUid());
		restorableItem.setLiveEntryUid(liveMailshare.uid);
		return liveMailshare;

	}

	@Override
	public void restoreEntryDependantContainers() {
		throw new UnsupportedOperationException("not implemented");
	}

	public void restoreMailbox() {
		ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(ITasksManager.class)
				.run(new RestoreBoxTask(backup, restorableItem.item(), mode));
	}
}
