/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2021
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.deserializer;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Path;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

import net.bluemind.dataprotect.service.internal.MailboxIndexJson;

public class CyrusSdsBackupMailbox implements AutoCloseable {
	protected static Logger logger = LoggerFactory.getLogger(CyrusSdsBackupMailbox.class);

	protected final JsonParser parser;
	protected String domainUid;
	protected String dataLocation;
	protected String mailboxUid;
	protected String kind;

	protected int version;
	protected List<CyrusSdsBackupFolder> folders = new ArrayList<>();
	protected Path storageFolder;
	protected MailboxIndexJson mailboxIndex;

	public CyrusSdsBackupMailbox(MailboxIndexJson mailboxIndex, Path workingFolder) throws Exception {
		this.mailboxIndex = mailboxIndex;
		storageFolder = workingFolder;
		parser = readJsonFileContent();
		parser.nextToken();
		try {
			readHeader();
		} catch (Exception e) {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			e.printStackTrace(pw);
			String errorMsg = sw.toString();
			throw new Exception("Error occurs trying to read " + mailboxIndex.mailboxUid() + "file from "
					+ workingFolder + " because:\r\n " + errorMsg);
		}
	}

	public CyrusSdsBackupMailbox(String path, Path workingFolder) throws Exception {
		this(new MailboxIndexJson(path, null), workingFolder);
	}

	protected JsonParser readJsonFileContent() throws IOException {
		return new JsonFactory().createParser(storageFolder.resolve(mailboxIndex.filename()).toFile());
	}

	private void readHeader() throws IOException, ParseException {
		if (parser.currentToken() != JsonToken.START_OBJECT) {
			throw new IllegalStateException("Expected an object");
		}
		while (parser.nextToken() != JsonToken.END_OBJECT) {
			String fieldName = parser.currentName();
			if (fieldName == null) {
				continue;
			}
			switch (fieldName) {
			case "domainUid":
				domainUid = parser.nextTextValue();
				break;
			case "dataLocation":
				dataLocation = parser.nextTextValue();
				break;
			case "version":
				version = parser.nextIntValue(1);
				break;
			case "mailboxUid":
				mailboxUid = parser.nextTextValue();
				break;
			case "kind":
				kind = parser.nextTextValue();
				break;
			case "folders":
				parser.nextToken();
				if (parser.currentToken() != JsonToken.START_ARRAY) {
					throw new IllegalStateException("Expected an array");
				}
				while (parser.nextToken() != JsonToken.END_ARRAY) {
					CyrusSdsBackupFolder folder = CyrusSdsBackupFolder.from(parser);
					if (folder != null) {
						folders.add(folder);
					}
				}
				return;
			case "backingstore":
				// Trash this, not used
				parser.nextToken();
				parser.skipChildren();
				break;
			default:
				parser.nextToken();
				break;
			}
		}
	}

	public String domainUid() {
		return domainUid;
	}

	public String dataLocation() {
		return dataLocation;
	}

	public int version() {
		return version;
	}

	public String mailboxUid() {
		return mailboxUid;
	}

	public String kind() {
		return kind;
	}

	public List<CyrusSdsBackupFolder> getFolders() {
		return folders;
	}

	@Override
	public void close() throws Exception {
		if (this.parser != null) {
			this.parser.close();
		}
	}
}