/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2021
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;

import io.netty.buffer.ByteBuf;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.tool.CoreCommonBackupWorker;
import net.bluemind.dataprotect.service.tool.ZipBuilder;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.Server;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.helper.ArchiveHelper;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class MailSdsWorker implements IBackupWorker {
	public static final Path WORKING_PATH = Paths.get("/var/spool/cyrus/data/dataprotect/sds");
	public static final Path OUTPUT_PATH = Paths.get("/var/backups/bluemind/sds");
	private static final Logger logger = LoggerFactory.getLogger(MailSdsWorker.class);

	@Override
	public String getDataType() {
		return WorkerDataType.SDS.value;
	}

	@Override
	public void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup) throws ServerFault {
		SystemConf sysconf = SysConfHelper.fromSharedMap();
		if (!ArchiveHelper.isSdsArchiveKind(sysconf)) {
			return;
		}
		List<String> domains = SPResolver.get().resolve(null).instance(IInCoreDomains.class).allUnfiltered().stream()
				.filter(d -> !"global.virt".equals(d.uid)).map(d -> d.uid).toList();
		cleanup(ctx, partGen, null);

		long startDate = Instant.now().getEpochSecond();
		try {
			if (!Files.exists(WORKING_PATH)) {
				Files.createDirectories(WORKING_PATH);
			}

			logger.info("Starting sds backup in {}", WORKING_PATH);
			ctx.info(String.format("Starting sds backup for %d domains: %s", domains.size(), domains));
			MailSdsBackupService backupService = new MailSdsBackupService(WORKING_PATH, partGen);
			backupService.setContext(ctx);
			backupService.backupDomains(domains, true);
			writeFilesToOutputPath(ctx, toBackup.value.address());
		} catch (IOException e) {
			partGen.withErrors = true;
			ctx.error(e, "Unable to create temporary directory '{}' for sds backup", WORKING_PATH);
		} finally {
			ctx.info(String.format("Ending sds backup in %d seconds", Instant.now().getEpochSecond() - startDate));
		}
	}

	private void writeFilesToOutputPath(IDPContext ctx, String backUpAddress) {
		INodeClient nc = NodeActivator.get(backUpAddress);
		nc.mkdirs(OUTPUT_PATH.toString());

		ctx.info(String.format("Start copiing SDS backup files from '%s' to '%s'", WORKING_PATH, OUTPUT_PATH));

		try {
			MailboxIndexJsonList indexJson = CoreCommonBackupWorker.readIndexJson(WORKING_PATH);
			List<ZipBuilder> zipBufferList = CoreCommonBackupWorker.createZipBufferList(
					indexJson.mailboxesIndexInfo().stream().map(i -> i.zipFileName).distinct().toList());
			Map<String, ByteBuf> jsonFiles = new HashMap<>();

			try (Stream<Path> stream = Files.list(WORKING_PATH)) {
				stream.filter(p -> !Files.isDirectory(p) && !CoreCommonBackupWorker.zipFileFilter(p))
						.forEach(p -> indexJson.mailboxesIndexInfo().stream()
								.filter(mailboxIndex -> mailboxIndex.filename.equals(p.getFileName().toString()))
								.findFirst() //
								.ifPresentOrElse(mailbox -> {
									Path jsonFilePath = WORKING_PATH.resolve(mailbox.filename);
									zipBufferList.stream()
											.filter(zipBuilder -> zipBuilder.zipFileName.equals(mailbox.zipFileName))
											.findFirst().ifPresent(zipFile -> {
												try {
													zipFile.addFileEntry(mailbox.filename, jsonFilePath);
												} catch (IOException e) {
													ctx.error(e, "Unable to write {} to zip {}", jsonFilePath,
															mailbox.zipFileName);
												}
											});
								}, () -> {
									try (InputStream instream = Files.newInputStream(p)) {
										ByteBuf jsonBuff = CoreCommonBackupWorker.mmapOutput();
										jsonBuff.writeBytes(instream.readAllBytes());
										jsonFiles.put(OUTPUT_PATH.resolve(p.getFileName()).toString(), jsonBuff);
									} catch (ServerFault | IOException e) {
										ctx.error(e, "Unable to copy {} to {}", WORKING_PATH.resolve(p.getFileName()),
												OUTPUT_PATH);
									}
								}));
			} catch (IOException e) {
				ctx.error(e, "Unable to list temporary directory files '{}' for sds backup", WORKING_PATH);
			} finally {
				CoreCommonBackupWorker.writeJsonFiles(ctx, jsonFiles);
				CoreCommonBackupWorker.writeZipFiles(ctx, OUTPUT_PATH, zipBufferList);
				ctx.info(String.format("sds backup done in '%s'", OUTPUT_PATH));
			}
		} catch (IOException ioe) {
			ctx.error(ioe, "Unable to read index.json from '{}' for sds backup", WORKING_PATH);
		}

	}

	@Override
	public Set<String> getDataDirs() {
		return Sets.newHashSet(OUTPUT_PATH.toString());
	}

	@Override
	public boolean supportsTag(String tag) {
		return CoreCommonBackupWorker.supportsTag(tag);
	}

}
