/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.mailbox.deserializer.RestoreSdsMailbox;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailshare.api.IMailshare;
import net.bluemind.mailshare.api.Mailshare;
import net.bluemind.sds.store.ISdsSyncStore;

public class MailshareBackupWorker extends MailBackupWorker {
	private static final String ZIP_FILE_PREFIX = "mailshare";

	public MailshareBackupWorker(IDPContext ctx, Path workingFolder, Path jsonIndex, RestoreSdsMailbox lastBackup,
			PartGeneration partGen) {
		super(ctx, workingFolder, jsonIndex, lastBackup, partGen);
	}

	@Override
	public MailboxIndexTotal partitionTreatment(String domainUid, String zipHash, List<String> partition) {
		List<MailboxIndexJson> mailsharesToBackup = new ArrayList<>();
		String zipFileName = CommonBackupWorker.zipFilename(ZIP_FILE_PREFIX, zipHash);
		partition.forEach(uid -> {
			MailboxIndexJson backupSdsMailshare = backupSdsMailshare(uid);
			if (backupSdsMailshare != null) {
				backupSdsMailshare.zipFileName(zipFileName);
				mailsharesToBackup.add(backupSdsMailshare);
			}
		});
		return new MailboxIndexTotal(mailsharesToBackup);
	}

	private MailboxIndexJson backupSdsMailshare(String uid) {
		DirEntry diruser = serviceProvider.instance(IDirectory.class, domain.uid).findByEntryUid(uid);
		return backupSdsMailshare(diruser, domain, downloadEmailContent);
	}

	public MailboxIndexJson backupSdsMailshare(DirEntry de, ItemValue<Domain> domain, boolean downloadEmailContent) {
		try {
			logger.info("backup single mailshare ({})", de.email != null ? de.email : de.displayName);

			IMailshare mailshareApi = serviceProvider.instance(IMailshare.class, domain.uid);
			ItemValue<Mailshare> mailshare = mailshareApi.getComplete(de.entryUid);
			String filename = String.format("mailshare_%s@%s.json", mailshare.value.name, domain.value.defaultAlias);
			Path outputPath = Paths.get(workingFolder.toAbsolutePath().toString(),
					String.format("mailshare_%s@%s.json", mailshare.value.name, domain.value.defaultAlias));

			Mailbox mailbox = mailshare.value.toMailbox();
			CyrusPartition part = CyrusPartition.forServerAndDomain(mailshare.value.dataLocation, domain.uid);
			ISdsSyncStore productionStore = productionStores.get(mailshare.value.dataLocation);

			IDbReplicatedMailboxes mailboxapi = serviceProvider.instance(IDbReplicatedMailboxes.class, part.name,
					mailshare.value.name);
			generateSdsMailboxJson(outputPath, domain, mailshare.uid, mailshare.value.name, mailbox, mailboxapi.all(),
					productionStore, downloadEmailContent);
			return new MailboxIndexJson(filename, mailshare.uid, domain.uid);
		} catch (Exception e) {
			logError(e, "Unable to backup mailshare {}", de.entryUid);
		}
		return null;
	}

}
