/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.directory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import net.bluemind.core.api.BMVersion;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerSubscriptionDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.api.GenerationIndex;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.directory.DirectoryBackupRepository.IndexedEntries;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.dataprotect.service.tool.CoreCommonBackupWorker;
import net.bluemind.dataprotect.service.tool.ZipBuilder;
import net.bluemind.device.api.Device;
import net.bluemind.device.api.IDevice;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.MailFilter;
import net.bluemind.mailbox.identity.api.IdentityDescription;
import net.bluemind.mailshare.api.IMailshare;
import net.bluemind.mailshare.api.Mailshare;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.resource.api.IResources;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.server.api.Server;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.Tag;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserMailIdentities;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.IUserSubscription;
import net.bluemind.user.api.User;
import net.bluemind.user.service.IInCoreUser;

public class DirectoryBackupWorker implements IBackupWorker {
	private static Logger logger = LoggerFactory.getLogger(DirectoryBackupWorker.class);
	public static final Path WORKING_PATH = ZipBuilder.WORKING_PATH.resolve("directory");
	private static final String ZIP_FILE_PREFIX = "directory";

	private static class DirEntryBackup extends CommonBackupWorker {
		private DirectoryBackupRepository repository;
		private List<GenerationIndex> indexedDirEntries = Collections.synchronizedList(new ArrayList<>());

		public DirEntryBackup(IDPContext ctx, Path p, PartGeneration partGen) {
			super(ctx, partGen);
			this.repository = new DirectoryBackupRepository(p);
		}

		@Override
		public List<MailboxIndexJson> partitionTreatment(String domainUid, String zipFile, List<String> partition) {
			String zipFileName = CommonBackupWorker.zipFilename(ZIP_FILE_PREFIX, zipFile);
			List<MailboxIndexJson> indexJson = new ArrayList<>();
			IDirectory directoryApi = provider().instance(IDirectory.class, domainUid);
			partition.forEach(uid -> {
				try {
					ItemValue<DirEntry> dirEntry = directoryApi.findItemValueByEntryUid(uid);
					switch (dirEntry.value.kind) {
					case USER -> {
						String backUpUserFile = backUpUser(domainUid, dirEntry);
						indexJson.add(new MailboxIndexJson(Arrays.asList(backUpUserFile), uid, domainUid, zipFileName));
					}
					case RESOURCE -> {
						String backUpResource = backUpResource(domainUid, dirEntry);
						indexJson.add(new MailboxIndexJson(Arrays.asList(backUpResource), uid, domainUid, zipFileName));
					}
					case GROUP -> {
						String backUpGroup = backUpGroup(domainUid, dirEntry);
						indexJson.add(new MailboxIndexJson(Arrays.asList(backUpGroup), uid, domainUid, zipFileName));
					}
					case MAILSHARE -> {
						String backUpMailshare = backUpMailshare(domainUid, dirEntry);
						indexJson
								.add(new MailboxIndexJson(Arrays.asList(backUpMailshare), uid, domainUid, zipFileName));
					}
					case SHARED_MAILBOX -> {
						String backUpSharedMailboxFile = backUpSharedMailbox(domainUid, dirEntry);
						indexJson.add(new MailboxIndexJson(Arrays.asList(backUpSharedMailboxFile), uid, domainUid,
								zipFileName));
					}
					case CALENDAR, ADDRESSBOOK -> backUpOthers(dirEntry);
					default -> logger.warn("Unsupported dirEntry type {} uid={}", dirEntry.value.kind, dirEntry.uid);
					}
				} catch (IOException e) {
					logError(e, "Unable to backup dirEntry {}", uid);
				}
			});
			indexJson.forEach(i -> i.zipFileName(zipFileName));
			indexJson.forEach(i -> i.domainUid(domainUid));
			return indexJson;
		}

		private MailboxIndexJsonList backupDomain(ItemValue<Domain> domain) {
			var q = DirEntryQuery.allWithHidden();
			ListResult<String> searchUids = provider().instance(IDirectory.class, domain.uid).searchUids(q);
			MailboxIndexJsonList ownerZipIndexDomain = runIt(domain.uid, searchUids.values);

			try {
				IndexedEntries rde = new IndexedEntries(BMVersion.getVersion(), indexedDirEntries);
				repository.writeDomainIndexedEntries(domain.uid, rde);
				indexedDirEntries.clear();
			} catch (IOException e) {
				logError(e, "Unable to backup IndexedEntries for domain {}", domain.uid);
			}

			return ownerZipIndexDomain;
		}

		private String backUpUser(String domainUid, ItemValue<DirEntry> dirEntry) throws IOException {
			IUser userApi = provider().instance(IUser.class, domainUid);
			ItemValue<User> user = userApi.getComplete(dirEntry.value.entryUid);
			RestorableUser restorableUser = new RestorableUser(dirEntry, user);

			IUserSettings userSettingsApi = provider().instance(IUserSettings.class, domainUid);
			restorableUser.settings = userSettingsApi.get(dirEntry.value.entryUid);
			restorableUser.subscriptions = getSubscriptions(domainUid, dirEntry);
			restorableUser.devices = getDevices(dirEntry);
			restorableUser.tags = getTags(dirEntry);
			restorableUser.identities = getIdentities(domainUid, dirEntry);
			restorableUser.credentials = getCredentials(domainUid, dirEntry);
			restorableUser.memberOf = getMemberOfGroup(dirEntry, userApi);
			restorableUser.filter = getMailFilter(domainUid, dirEntry);
			restorableUser.item.value.password = getPassword(domainUid, restorableUser.item.internalId);
			return repository.writeUser(restorableUser, indexedDirEntries);
		}

		private String backUpSharedMailbox(String domainUid, ItemValue<DirEntry> dirEntry) throws IOException {
			IUser sharedMailboxApi = provider().instance(IUser.class, domainUid);
			ItemValue<User> sharedMailbox = sharedMailboxApi.getComplete(dirEntry.value.entryUid);
			RestorableUser restorableSharedMailbox = new RestorableUser(dirEntry, sharedMailbox);

			IUserSettings sharedMailboxSettingsApi = provider().instance(IUserSettings.class, domainUid);
			restorableSharedMailbox.settings = sharedMailboxSettingsApi.get(dirEntry.value.entryUid);
			restorableSharedMailbox.tags = getTags(dirEntry);
			restorableSharedMailbox.memberOf = getMemberOfGroup(dirEntry, sharedMailboxApi);
			restorableSharedMailbox.identities = getIdentities(domainUid, dirEntry);
			restorableSharedMailbox.filter = getMailFilter(domainUid, dirEntry);
			return repository.writeUser(restorableSharedMailbox, indexedDirEntries);
		}

		private String backUpResource(String domainUid, ItemValue<DirEntry> dirEntry) throws IOException {
			IResources resourceApi = provider().instance(IResources.class, domainUid);
			ItemValue<ResourceDescriptor> resource = resourceApi.getComplete(dirEntry.value.entryUid);
			return repository.writeResource(dirEntry, resource, indexedDirEntries);
		}

		private String backUpGroup(String domainUid, ItemValue<DirEntry> dirEntry) throws IOException {
			IGroup groupApi = provider().instance(IGroup.class, domainUid);
			List<Member> members = groupApi.getMembers(dirEntry.value.entryUid);
			ItemValue<Group> group = groupApi.getComplete(dirEntry.value.entryUid);
			return repository.writeGroup(dirEntry, group, members, indexedDirEntries);
		}

		private String backUpMailshare(String domainUid, ItemValue<DirEntry> dirEntry) throws IOException {
			IMailshare mailshareApi = provider().instance(IMailshare.class, domainUid);
			ItemValue<Mailshare> mailshare = mailshareApi.getComplete(dirEntry.value.entryUid);
			return repository.writeMailshare(dirEntry, mailshare, indexedDirEntries);
		}

		private void backUpOthers(ItemValue<DirEntry> dirEntry) {
			repository.writeDirEntry(dirEntry.value, indexedDirEntries);
		}

		private List<Credential> getCredentials(String domainUid, ItemValue<DirEntry> dirEntry) {
			try {
				return provider().instance(IInternalCredentials.class, domainUid)
						.getUserCredentials(dirEntry.value.entryUid);
			} catch (Exception e) {
				logger.error("Unable to backup credentials for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		private String getPassword(String domainUid, long userId) {
			return provider().instance(IInCoreUser.class, domainUid).getPassword(userId);
		}

		private MailFilter getMailFilter(String domainUid, ItemValue<DirEntry> dirEntry) {
			return provider().instance(IMailboxes.class, domainUid).getMailboxFilter(dirEntry.uid);
		}

		private List<ItemValue<Group>> getMemberOfGroup(ItemValue<DirEntry> dirEntry, IUser userApi) {
			try {
				return userApi.memberOf(dirEntry.value.entryUid);
			} catch (Exception e) {
				logger.error("Unable to backup members of group for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		private List<IdentityDescription> getIdentities(String domainUid, ItemValue<DirEntry> dirEntry) {
			try {
				return provider().instance(IUserMailIdentities.class, domainUid, dirEntry.value.entryUid)
						.getIdentities();
			} catch (Exception e) {
				logger.error("Unable to backup user identities for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		private List<ItemValue<Tag>> getTags(ItemValue<DirEntry> dirEntry) {
			try {
				return provider().instance(ITags.class, ITagUids.defaultTags(dirEntry.value.entryUid)).all();
			} catch (Exception e) {
				logger.error("Unable to backup tags for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		private List<ItemValue<Device>> getDevices(ItemValue<DirEntry> dirEntry) {
			try {
				return provider().instance(IDevice.class, dirEntry.value.entryUid).list().values;
			} catch (Exception e) {
				logger.error("Unable to backup devices for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		private List<ContainerSubscriptionDescriptor> getSubscriptions(String domainUid, ItemValue<DirEntry> dirEntry) {
			try {
				return provider().instance(IUserSubscription.class, domainUid)
						.listSubscriptions(dirEntry.value.entryUid, null).stream()
						.filter(csd -> !csd.owner.equals(dirEntry.value.entryUid)).toList();
			} catch (Exception e) {
				logger.error("Unable to backup subscriptions for {}", dirEntry);
			}
			return Collections.emptyList();
		}

		public void backupAll() {
			var domainsApi = provider().instance(IInCoreDomains.class);
			List<MailboxIndexJson> domainsZip = new ArrayList<>();
			domainsApi.allUnfiltered().stream().filter(d -> !"global.virt".equals(d.uid)).forEach(domain -> {
				logger.info("Backup directories for domain {}", domain.value.defaultAlias);
				domainsZip.addAll(backupDomain(domain).mailboxesIndexInfo());
			});

			Path rootPath = repository.repositoryPath.rootPath();
			try {
				RepositoryBackupPath.writeIndexFileToDir(rootPath, domainsZip);
			} catch (IOException e) {
				logError(e, "Error trying to write index.json {}", rootPath);
			}
		}

	}

	@Override
	public void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup) throws ServerFault {
		cleanup(ctx, partGen, null);
		Path outputFolder = DirectoryBackupRepository.DEFAULT_PATH;
		long startDate = Instant.now().getEpochSecond();

		try {
			if (!Files.exists(WORKING_PATH)) {
				Files.createDirectories(WORKING_PATH);
			}

			logger.info("Starting direntries backup in {}", WORKING_PATH);
			ctx.info("Starting direntries backup");
			new DirEntryBackup(ctx, WORKING_PATH, partGen).backupAll();
			prepareDirectoryDataDirs(ctx, toBackup, WORKING_PATH, outputFolder);
		} catch (IOException e) {
			partGen.withErrors = true;
			ctx.error(e, "Unable to create temporary directory for directory backup");
		} finally {
			ctx.info(String.format("Ending direntries backup in %d seconds",
					Instant.now().getEpochSecond() - startDate));
		}
	}

	private static void prepareDirectoryDataDirs(IDPContext ctx, ItemValue<Server> toBackup, Path tmpFolder,
			Path outputFolder) throws IOException {
		INodeClient nc = NodeActivator.get(toBackup.value.address());
		nc.mkdirs(outputFolder.toString());
		if (!Files.exists(outputFolder)) {
			Files.createDirectories(outputFolder);
		}

		MailboxIndexJsonList indexJson = CoreCommonBackupWorker.readIndexJson(tmpFolder);
		if (indexJson == null) {
			ctx.error("Unable to read index.json from '{}' for backup", tmpFolder.toString());
			return;
		}

		Map<String, ByteBuf> jsonFiles = new HashMap<>();
		List<ZipBuilder> zipBufferList = CoreCommonBackupWorker.createZipBufferList(
				indexJson.mailboxesIndexInfo().stream().map(i -> i.zipFileName).distinct().toList());
		try {
			indexJson.mailboxesIndexInfo().forEach(mailbox -> {
				try {
					CoreCommonBackupWorker.writeMailboxIndexJson(ctx, mailbox, zipBufferList, tmpFolder);
				} catch (FileNotFoundException e) {
					ctx.error(e, e.getMessage());
				}
			});

			CoreCommonBackupWorker.prepareIndexJson(tmpFolder, outputFolder, jsonFiles);

			try (Stream<Path> stream = Files.list(tmpFolder)) {
				stream.filter(p -> !Files.isDirectory(p) && p.getFileName().toString().startsWith("___")
						&& p.getFileName().toString().endsWith(".json")).forEach(p -> {
							Path relativePath = tmpFolder.relativize(p);
							Path targetPath = outputFolder.resolve(relativePath);
							try (InputStream instream = Files.newInputStream(p);
									OutputStream outStream = new FileOutputStream(new File(targetPath.toString()))) {
								outStream.write(instream.readAllBytes());
							} catch (ServerFault | IOException e) {
								logger.error("Unable to copy {} to {}: {}", p, targetPath, e.getMessage());
								ctx.error(e, "Unable to copy {} to {}", p, targetPath);
							}
						});
			}
		} finally {
			CoreCommonBackupWorker.writeJsonFiles(ctx, jsonFiles);
			CoreCommonBackupWorker.writeZipFiles(ctx, outputFolder, zipBufferList);
		}
	}

	@Override
	public String getDataType() {
		return WorkerDataType.DIRECTORY.value;
	}

	@Override
	public Set<String> getDataDirs() {
		return Set.of(DirectoryBackupRepository.DEFAULT_PATH.toString());
	}

	@Override
	public boolean supportsTag(String tag) {
		return CoreCommonBackupWorker.supportsTag(tag);
	}
}
